<?php
declare(strict_types=1);

namespace NewUploader;

/**
 * فاساد ذخیره‌سازی – پشت‌صحنه می‌تواند JSON یا MySQL باشد.
 * دیفالت: JSON
 */
class Storage
{
    private static ?Storage $instance = null;

    private StorageDriverInterface $driver;

    private function __construct()
    {
        $this->driver = $this->createDriverFromConfig();
    }

    public static function instance(): Storage
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function createDriverFromConfig(): StorageDriverInterface
    {
        $baseDir  = dirname(__DIR__);
        $confPath = $baseDir . '/storage/storage_config.json';

        $driverName = 'json'; // پیش‌فرض

        if (is_file($confPath)) {
            $raw = file_get_contents($confPath);
            $cfg = json_decode($raw, true);
            if (is_array($cfg) && !empty($cfg['driver'])) {
                $driverName = $cfg['driver'];
            }
        }

        if ($driverName === 'mysql' && isset($GLOBALS['connect']) && $GLOBALS['connect'] instanceof \mysqli) {
            return new StorageMysql($GLOBALS['connect']);
        }

        // اگر mysql انتخاب شده ولی اتصال موجود نیست، برمی‌گردیم به json
        return new StorageJson($baseDir . '/storage');
    }

    // --- متدهای قابل استفاده در کل ربات ---

    public function ensureUser(int $id, ?string $name = null): void
    {
        $this->driver->ensureUser($id, $name);
    }

    public function getUserStep(int $id): string
    {
        return $this->driver->getUserStep($id);
    }

    public function setUserStep(int $id, string $step): void
    {
        $this->driver->setUserStep($id, $step);
    }

    public function generateUniqueFileCode(int $length = 6): string
    {
        return $this->driver->generateUniqueFileCode($length);
    }

    /**
     * ذخیره فایل اصلی
     */
    public function saveFile(array $data): void
    {
        $this->driver->saveFile($data);
    }

    /**
     * نگه‌داشتن نگاشت فایل برای کاربر (userfiles)
     */
    public function saveUserFile(int $userId, string $code, string $fileId, string $caption, string $type): void
    {
        $this->driver->saveUserFile($userId, $code, $fileId, $caption, $type);
    }
}

interface StorageDriverInterface
{
    public function ensureUser(int $id, ?string $name = null): void;

    public function getUserStep(int $id): string;

    public function setUserStep(int $id, string $step): void;

    public function generateUniqueFileCode(int $length = 6): string;

    public function saveFile(array $data): void;

    public function saveUserFile(int $userId, string $code, string $fileId, string $caption, string $type): void;
}


