<?php

declare(strict_types=1);

namespace Bot\Tests\Unit;

use Bot\Domain\Entity\User;
use Bot\Domain\Repository\UserRepositoryInterface;
use Bot\Domain\Service\UserService;
use Bot\Infrastructure\Repository\InMemoryUserRepository;
use PHPUnit\Framework\TestCase;

final class UserServiceTest extends TestCase
{
    private UserRepositoryInterface $userRepository;
    private UserService $userService;

    protected function setUp(): void
    {
        $this->userRepository = new InMemoryUserRepository();
        $this->userService = new UserService($this->userRepository);
    }

    public function testGetOrCreateByTelegramIdCreatesNewUser(): void
    {
        $telegramId = 12345;
        $user = $this->userService->getOrCreateByTelegramId($telegramId);

        $this->assertInstanceOf(User::class, $user);
        $this->assertEquals($telegramId, $user->getTelegramId());
        $this->assertEquals('free', $user->getMode());
    }

    public function testGetOrCreateByTelegramIdReturnsExistingUser(): void
    {
        $telegramId = 12345;
        $existingUser = new User(1, $telegramId, 'premium');
        $this->userRepository->save($existingUser);

        $user = $this->userService->getOrCreateByTelegramId($telegramId);

        $this->assertSame($existingUser, $user);
        $this->assertEquals('premium', $user->getMode());
    }

    public function testUpgradeToPremium(): void
    {
        $telegramId = 12345;
        $user = $this->userService->getOrCreateByTelegramId($telegramId);
        $this->assertEquals('free', $user->getMode());

        $this->userService->upgradeToPremium($user);

        $this->assertEquals('premium', $user->getMode());
    }

    public function testGetAllReturnsAllUsers(): void
    {
        $this->userService->getOrCreateByTelegramId(111);
        $this->userService->getOrCreateByTelegramId(222);
        $this->userService->getOrCreateByTelegramId(333);

        $allUsers = $this->userService->getAll();

        $this->assertCount(3, $allUsers);
    }
}

