<?php

declare(strict_types=1);

namespace Bot\Tests\Unit;

use Bot\Domain\Entity\Media;
use Bot\Domain\Repository\MediaRepositoryInterface;
use Bot\Domain\Service\MediaService;
use Bot\Infrastructure\Repository\InMemoryMediaRepository;
use PHPUnit\Framework\TestCase;

final class MediaServiceTest extends TestCase
{
    private MediaRepositoryInterface $mediaRepository;
    private MediaService $mediaService;

    protected function setUp(): void
    {
        $this->mediaRepository = new InMemoryMediaRepository();
        $this->mediaService = new MediaService($this->mediaRepository);
    }

    public function testCreateMedia(): void
    {
        $folderId = 1;
        $ownerAdminId = 10;
        $telegramFileId = 'file123';
        $type = 'photo';
        $caption = 'Test caption';

        $media = $this->mediaService->createMedia(
            $folderId,
            $ownerAdminId,
            $telegramFileId,
            $type,
            $caption
        );

        $this->assertInstanceOf(Media::class, $media);
        $this->assertEquals($folderId, $media->getFolderId());
        $this->assertEquals($ownerAdminId, $media->getOwnerAdminId());
        $this->assertEquals($telegramFileId, $media->getTelegramFileId());
        $this->assertEquals($type, $media->getType());
        $this->assertEquals($caption, $media->getCaption());
    }

    public function testCreateMediaWithThumbnail(): void
    {
        $media = $this->mediaService->createMedia(
            1,
            10,
            'file123',
            'video',
            'Video caption',
            'thumb123'
        );

        $this->assertEquals('thumb123', $media->getThumbnailFileId());
        $this->assertTrue($media->isThumbnailEnabled());
    }

    public function testListByFolder(): void
    {
        $folderId1 = 1;
        $folderId2 = 2;

        $this->mediaService->createMedia($folderId1, 10, 'file1', 'photo');
        $this->mediaService->createMedia($folderId1, 10, 'file2', 'photo');
        $this->mediaService->createMedia($folderId2, 10, 'file3', 'photo');

        $mediaInFolder1 = $this->mediaService->listByFolder($folderId1);
        $this->assertCount(2, $mediaInFolder1);

        $mediaInFolder2 = $this->mediaService->listByFolder($folderId2);
        $this->assertCount(1, $mediaInFolder2);
    }

    public function testUpdateMedia(): void
    {
        $media = $this->mediaService->createMedia(1, 10, 'file1', 'photo', 'Original caption');
        $this->assertEquals('Original caption', $media->getCaption());

        // گرفتن media با ID واقعی از repository
        $mediaList = $this->mediaService->listByFolder(1);
        $this->assertNotEmpty($mediaList);
        $savedMedia = $mediaList[0];

        $savedMedia->setCaption('Updated caption');
        $savedMedia->setAutoDeleteSeconds(300);
        $this->mediaService->updateMedia($savedMedia);

        $updatedMedia = $this->mediaService->findById($savedMedia->getId());
        $this->assertNotNull($updatedMedia);
        $this->assertEquals('Updated caption', $updatedMedia->getCaption());
        $this->assertEquals(300, $updatedMedia->getAutoDeleteSeconds());
    }

    public function testToggleThumbnail(): void
    {
        $media = $this->mediaService->createMedia(1, 10, 'file1', 'photo');
        $this->assertTrue($media->isThumbnailEnabled());

        // گرفتن media با ID واقعی از repository
        $mediaList = $this->mediaService->listByFolder(1);
        $this->assertNotEmpty($mediaList);
        $savedMedia = $mediaList[0];

        $savedMedia->toggleThumbnail(false);
        $this->mediaService->updateMedia($savedMedia);

        $updatedMedia = $this->mediaService->findById($savedMedia->getId());
        $this->assertNotNull($updatedMedia);
        $this->assertFalse($updatedMedia->isThumbnailEnabled());
    }
}

