<?php

declare(strict_types=1);

use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\AnalyticsService;
use Bot\Domain\Service\ForcedJoinService;
use Bot\Domain\Service\LogService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\UserService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use SergiX44\Nutgram\Nutgram;

/**
 * هندلر /start برای نمایش منوی اصلی و پردازش لینک‌های یکتا
 */
function registerStartHandler(
    Nutgram $bot,
    UserService $userService,
    ForcedJoinService $forcedJoinService,
    AnalyticsService $analyticsService,
    InMemoryLinkStore $linkStore,
    MediaService $mediaService,
    AdminService $adminService,
    array &$downloadCounter,
    int $freeDownloadLimit,
    ?string $forcedJoinChannel,
    ?int $ownerTelegramId,
    LogService $logService,
): void {
    $bot->onCommand('start', function (Nutgram $bot) use (
        $userService,
        $forcedJoinService,
        $analyticsService,
        $linkStore,
        $mediaService,
        &$downloadCounter,
        $freeDownloadLimit,
        $forcedJoinChannel,
        $adminService,
        $ownerTelegramId,
        $logService,
    ) {
        $from = $bot->user();
        if ($from === null) {
            return;
        }

        $telegramId = (int) $from->id;

        // بررسی payload برای لینک یکتا
        $message = $bot->message();
        $text    = $message?->text ?? '';
        $parts   = explode(' ', $text, 2);
        $payload = isset($parts[1]) ? trim($parts[1]) : '';

        // اگر payload وجود دارد، لینک یکتا است
        if ($payload !== '') {
            // اگر payload با get_ شروع نشده باشد، اضافه می‌کنیم (برای سازگاری)
            if (!str_starts_with($payload, 'get_')) {
                $payload = 'get_' . $payload;
            }
            
            // ابتدا تلاش می‌کنیم UUID را استخراج کنیم (لینک جدید)
            $uuid = $linkStore->extractUuid($payload);
            $media = null;
            
            if ($uuid !== null) {
                // استفاده از UUID برای پیدا کردن مدیا
                $media = $mediaService->findByUuid($uuid);
                
                $logService->debug('استخراج UUID از لینک', [
                    'telegram_id' => $telegramId,
                    'payload' => $payload,
                    'uuid' => $uuid,
                    'found' => $media !== null,
                ], $telegramId);
            } else {
                // اگر UUID پیدا نشد، تلاش می‌کنیم media_id را استخراج کنیم (لینک قدیمی)
                $mediaId = $linkStore->extractMediaId($payload);
                
                if ($mediaId !== null) {
                    $media = $mediaService->findById($mediaId);
                    
                    $logService->debug('استخراج media_id از لینک قدیمی', [
                        'telegram_id' => $telegramId,
                        'payload' => $payload,
                        'media_id' => $mediaId,
                        'found' => $media !== null,
                    ], $telegramId);
                } else {
                    $logService->warning('لینک یکتا نامعتبر است', [
                        'telegram_id' => $telegramId,
                        'payload' => $payload,
                    ], $telegramId);
                    $bot->sendMessage("لینک یکتای مدیا نامعتبر است.");
                    return;
                }
            }
            
            if ($media === null) {
                $logService->warning('مدیا یافت نشد', [
                    'telegram_id' => $telegramId,
                    'payload' => $payload,
                ], $telegramId);
                $bot->sendMessage("مدیا یافت نشد یا حذف شده است.");
                return;
            }
            
            // لاگ برای دیباگ
            $logService->info('دریافت لینک یکتا', [
                'telegram_id' => $telegramId,
                'payload' => $payload,
                'media_id' => $media->getId(),
                'uuid' => $media->getUuid(),
                'file_id' => $media->getTelegramFileId(),
                'type' => $media->getType(),
            ], $telegramId);

            // Forced-join: در صورت تنظیم کانال، عضویت بررسی می‌شود
            $hasJoinedRequiredChannels = true;
            if ($forcedJoinChannel !== null && $forcedJoinChannel !== '') {
                try {
                    $member = $bot->getChatMember($forcedJoinChannel, $telegramId);
                    if ($member === null || in_array($member->status, ['left', 'kicked'], true)) {
                        $hasJoinedRequiredChannels = false;
                    }
                } catch (Throwable $e) {
                    SimpleLogger::error('خطا در بررسی عضویت کانال', [
                        'telegram_id' => $telegramId,
                        'channel' => $forcedJoinChannel,
                        'error' => $e->getMessage(),
                    ]);
                    $hasJoinedRequiredChannels = false;
                }
            }

            if (!$forcedJoinService->isUserAllowedToDownload($hasJoinedRequiredChannels)) {
                $text = "برای دریافت مدیا، ابتدا باید در کانال(های) مورد نیاز عضو شوید.";
                if ($forcedJoinChannel !== null) {
                    $text .= PHP_EOL . $forcedJoinChannel;
                }

                $bot->sendMessage($text);
                return;
            }

            // کاربر را ایجاد/بازیابی می‌کنیم
            $user = $userService->getOrCreateByTelegramId($telegramId);

            // محدودیت ساده برای حالت رایگان
            $current = $downloadCounter[$telegramId] ?? 0;
            if ($user->getMode() === 'free' && $current >= $freeDownloadLimit) {
                $bot->sendMessage("سهمیه دانلود شما در حالت رایگان تمام شده است. برای استفاده نامحدود، حساب خود را به پریمیوم ارتقا دهید.");
                return;
            }

            $downloadCounter[$telegramId] = $current + 1;

            // ارسال مدیا بر اساس نوع ثبت‌شده در مدیا
            $caption = $media->getCaption();
            
            $logService->debug('در حال ارسال مدیا', [
                'telegram_id' => $telegramId,
                'media_id' => $media->getId(),
                'type' => $media->getType(),
            ], $telegramId);
            
            try {
                switch ($media->getType()) {
                    case 'photo':
                        $bot->sendPhoto($media->getTelegramFileId(), caption: $caption);
                        $logService->info('Photo ارسال شد', [
                            'telegram_id' => $telegramId,
                            'media_id' => $media->getId(),
                        ], $telegramId);
                        break;
                    case 'video':
                        $bot->sendVideo($media->getTelegramFileId(), caption: $caption);
                        $logService->info('Video ارسال شد', [
                            'telegram_id' => $telegramId,
                            'media_id' => $media->getId(),
                        ], $telegramId);
                        break;
                    case 'document':
                    default:
                        $bot->sendDocument($media->getTelegramFileId(), caption: $caption);
                        $logService->info('Document ارسال شد', [
                            'telegram_id' => $telegramId,
                            'media_id' => $media->getId(),
                        ], $telegramId);
                        break;
                }
            } catch (Throwable $e) {
                $logService->error('خطا در ارسال مدیا', [
                    'telegram_id' => $telegramId,
                    'media_id' => $media->getId(),
                    'file_id' => $media->getTelegramFileId(),
                    'type' => $media->getType(),
                ], $telegramId, exception: $e);
                $bot->sendMessage("خطا در ارسال فایل. لطفاً دوباره تلاش کنید یا با پشتیبانی تماس بگیرید.");
                return;
            }

            // ثبت آنالیتیکس ساده
            try {
                $analyticsService->recordDownload($user->getId(), $media->getId());
                $logService->info('دانلود مدیا موفق', [
                    'user_id' => $user->getId(),
                    'telegram_id' => $telegramId,
                    'media_id' => $media->getId(),
                    'media_type' => $media->getType(),
                ], $telegramId);
            } catch (Throwable $e) {
                $logService->error('خطا در ثبت آنالیتیکس دانلود', [
                    'user_id' => $user->getId(),
                    'media_id' => $media->getId(),
                ], $telegramId, exception: $e);
            }
            
            return;
        }

        // شروع معمولی - نمایش منوی اصلی
        $user = $userService->getOrCreateByTelegramId($telegramId);
        
        // اگر owner است، به صورت خودکار در دیتابیس ذخیره می‌شود
        if ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) {
            $adminService->getOrCreateOwner($telegramId);
        }
        
        $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
            || $adminService->getByTelegramId($telegramId) !== null;
        
        $modeText = $user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان';
        $welcomeText = "👋 سلام! به ربات آپلودر خوش آمدید." . PHP_EOL . PHP_EOL;
        $welcomeText .= "وضعیت حساب: {$modeText}";
        
        $bot->sendMessage(
            $welcomeText,
            reply_markup: buildMainMenu($isAdmin)
        );
    });
}


