<?php

declare(strict_types=1);

use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\AnalyticsService;
use Bot\Domain\Service\ForcedJoinService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\PaymentService;
use Bot\Domain\Service\UserService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use Bot\Infrastructure\Logging\SimpleLogger;
use Bot\Infrastructure\Payment\NextPayGateway;
use Bot\Infrastructure\Payment\ZarinPalGateway;
use Bot\Infrastructure\Repository\InMemoryAdminRepository;
use Bot\Infrastructure\Repository\InMemoryAnalyticsRepository;
use Bot\Infrastructure\Repository\InMemoryMediaRepository;
use Bot\Infrastructure\Repository\InMemoryPaymentRepository;
use Bot\Infrastructure\Repository\InMemoryUserRepository;
use SergiX44\Nutgram\Telegram\Types\Payment\LabeledPrice;
use Dotenv\Dotenv;
use SergiX44\Nutgram\Nutgram;

require __DIR__ . '/../vendor/autoload.php';

// Load environment variables from .env if present
$dotenv = Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->load();

$botToken = $_ENV['BOT_TOKEN'] ?? null;
if ($botToken === null || $botToken === '') {
    throw new RuntimeException('BOT_TOKEN env variable is required.');
}

$forcedJoinChannel = $_ENV['FORCED_JOIN_CHANNEL'] ?? null; // مثال: @your_channel
$ownerTelegramId   = isset($_ENV['OWNER_TELEGRAM_ID'])
    ? (int) $_ENV['OWNER_TELEGRAM_ID']
    : null;

// Payment gateway configuration
$nextpayApiKey = $_ENV['NEXTPAY_API_KEY'] ?? '';
$zarinpalMerchantId = $_ENV['ZARINPAL_MERCHANT_ID'] ?? '';
$zarinpalSandbox = isset($_ENV['ZARINPAL_SANDBOX']) && $_ENV['ZARINPAL_SANDBOX'] === 'true';
$paymentCallbackBaseUrl = $_ENV['PAYMENT_CALLBACK_BASE_URL'] ?? 'https://yourdomain.com/payment_callback.php';

// ریپوزیتوری‌ها (in-memory برای نسخه‌ی مینیمال)
$userRepo       = new InMemoryUserRepository();
$adminRepo      = new InMemoryAdminRepository();
$mediaRepo      = new InMemoryMediaRepository();
$analyticsRepo  = new InMemoryAnalyticsRepository();
$paymentRepo    = new InMemoryPaymentRepository();
$adsRepo        = new \Bot\Infrastructure\Repository\InMemoryAdsRepository();
$linkStore      = new InMemoryLinkStore();

// سرویس‌های دامین
$userService       = new UserService($userRepo);
$adminService      = new AdminService($adminRepo);
$mediaService      = new MediaService($mediaRepo);
$analyticsService  = new AnalyticsService($analyticsRepo);
$paymentService    = new PaymentService($paymentRepo);
$adsService        = new \Bot\Domain\Service\AdsService($adsRepo);
$forcedJoinService = new ForcedJoinService();

// شمارنده‌ی ساده برای محدودیت حالت رایگان
/** @var array<int,int> $downloadCounter keyed by telegram_id */
$downloadCounter = [];
$freeDownloadLimit = 3;

// State management برای ورودی‌های کاربر
/** @var array<int,array{action:string,step?:int,data?:array}> $userStates keyed by telegram_id */
$userStates = [];

$bot = new Nutgram($botToken);

// ==================== توابع Helper برای منوها ====================

/**
 * ساخت منوی اصلی (برای کاربران و ادمین‌ها)
 */
function buildMainMenu(bool $isAdmin): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if ($isAdmin) {
        $keyboard->addRow(
            InlineKeyboardButton::make('🔧 پنل ادمین', callback_data: 'menu:admin')
        );
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⭐ ارتقا به پریمیوم', callback_data: 'menu:premium')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📊 آمار شخصی', callback_data: 'menu:user:stats')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی ادمین
 */
function buildAdminMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📁 مدیریت مدیا', callback_data: 'menu:admin:media')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📢 مدیریت تبلیغات', callback_data: 'menu:admin:ads')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('👥 مدیریت ادمین‌ها', callback_data: 'menu:admin:admins')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📊 آمار و گزارشات', callback_data: 'menu:admin:stats')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 برودکست', callback_data: 'menu:admin:broadcast')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی پرداخت
 */
function buildPremiumMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⭐ Telegram Stars', callback_data: 'action:premium:stars')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('💳 NextPay', callback_data: 'action:premium:nextpay')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('💳 ZarinPal', callback_data: 'action:premium:zarinpal')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت مدیا
 */
function buildMediaMenu(int $mediaId = 0): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if ($mediaId > 0) {
        $keyboard->addRow(
            InlineKeyboardButton::make('ℹ️ اطلاعات مدیا', callback_data: "action:media:info:{$mediaId}")
        );
        
        $keyboard->addRow(
            InlineKeyboardButton::make('🖼️ Toggle Thumbnail', callback_data: "action:media:toggle_thumb:{$mediaId}"),
            InlineKeyboardButton::make('🔗 Toggle Link Caption', callback_data: "action:media:toggle_link:{$mediaId}")
        );
        
        $keyboard->addRow(
            InlineKeyboardButton::make('⏱️ Auto-Delete', callback_data: "action:media:autodelete:{$mediaId}")
        );
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت تبلیغات
 */
function buildAdsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('➕ افزودن تبلیغ', callback_data: 'action:ads:create:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لیست تبلیغات', callback_data: 'action:ads:list')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت مدیا (با لیست مدیاها)
 */
function buildMediaListMenu(array $mediaList, int $page = 0, int $perPage = 5): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if (empty($mediaList)) {
        $keyboard->addRow(
            InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
        );
        return $keyboard;
    }
    
    $start = $page * $perPage;
    $end = min($start + $perPage, count($mediaList));
    $mediaPage = array_slice($mediaList, $start, $perPage);
    
    foreach ($mediaPage as $media) {
        $keyboard->addRow(
            InlineKeyboardButton::make(
                "📁 #{$media->getId()} - {$media->getType()}",
                callback_data: "action:media:select:{$media->getId()}"
            )
        );
    }
    
    // دکمه‌های navigation
    $navRow = [];
    if ($page > 0) {
        $navRow[] = InlineKeyboardButton::make('◀️ قبلی', callback_data: "action:media:list:page:" . ($page - 1));
    }
    if ($end < count($mediaList)) {
        $navRow[] = InlineKeyboardButton::make('بعدی ▶️', callback_data: "action:media:list:page:" . ($page + 1));
    }
    if (!empty($navRow)) {
        $keyboard->addRow(...$navRow);
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت یک مدیا خاص
 */
function buildMediaDetailMenu(int $mediaId): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('ℹ️ اطلاعات مدیا', callback_data: "action:media:info:{$mediaId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🖼️ Toggle Thumbnail', callback_data: "action:media:toggle_thumb:{$mediaId}"),
        InlineKeyboardButton::make('🔗 Toggle Link Caption', callback_data: "action:media:toggle_link:{$mediaId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⏱️ تنظیم Auto-Delete', callback_data: "action:media:autodelete:input:{$mediaId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت به لیست', callback_data: 'action:media:list:page:0')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی برودکست
 */
function buildBroadcastMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 ارسال به همه کاربران', callback_data: 'action:broadcast:input:all')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 ارسال به کانال/گروه', callback_data: 'action:broadcast:input:channel')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت ادمین‌ها
 */
function buildAdminsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('➕ افزودن ادمین', callback_data: 'action:admin:add:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🗑️ حذف ادمین', callback_data: 'action:admin:remove:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لیست ادمین‌ها', callback_data: 'action:admin:list')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی آمار ادمین
 */
function buildAdminStatsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📊 آمار کلی', callback_data: 'action:stats:general')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('💰 گزارش درآمد', callback_data: 'action:stats:revenue')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

// ==================== هندلر منوها ====================

// هندلر /start برای نمایش منوی اصلی
$bot->onCommand('start', function (Nutgram $bot) use (
    $userService,
    $forcedJoinService,
    $analyticsService,
    $linkStore,
    &$downloadCounter,
    $freeDownloadLimit,
    $forcedJoinChannel,
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // بررسی payload برای لینک یکتا
    $message = $bot->message();
    $text    = $message?->text ?? '';
    $parts   = explode(' ', $text, 2);
    $payload = isset($parts[1]) ? trim($parts[1]) : '';

    // اگر payload وجود دارد، لینک یکتا است
    if ($payload !== '') {
        $linkData = $linkStore->resolve($payload);
        if ($linkData === null) {
            $bot->sendMessage("لینک یکتای مدیا یافت نشد یا منقضی شده است.");
            return;
        }

        // Forced-join: در صورت تنظیم کانال، عضویت بررسی می‌شود
        $hasJoinedRequiredChannels = true;
        if ($forcedJoinChannel !== null && $forcedJoinChannel !== '') {
            try {
                $member = $bot->getChatMember($forcedJoinChannel, $telegramId);
                if ($member === null || in_array($member->status, ['left', 'kicked'], true)) {
                    $hasJoinedRequiredChannels = false;
                }
            } catch (Throwable $e) {
                SimpleLogger::error('خطا در بررسی عضویت کانال', [
                    'telegram_id' => $telegramId,
                    'channel' => $forcedJoinChannel,
                    'error' => $e->getMessage(),
                ]);
                $hasJoinedRequiredChannels = false;
            }
        }

        if (!$forcedJoinService->isUserAllowedToDownload($hasJoinedRequiredChannels)) {
            $text = "برای دریافت مدیا، ابتدا باید در کانال(های) مورد نیاز عضو شوید.";
            if ($forcedJoinChannel !== null) {
                $text .= PHP_EOL . $forcedJoinChannel;
            }

            $bot->sendMessage($text);
            return;
        }

        // کاربر را ایجاد/بازیابی می‌کنیم
        $user = $userService->getOrCreateByTelegramId($telegramId);

        // محدودیت ساده برای حالت رایگان
        $current = $downloadCounter[$telegramId] ?? 0;
        if ($user->getMode() === 'free' && $current >= $freeDownloadLimit) {
            $bot->sendMessage("سهمیه دانلود شما در حالت رایگان تمام شده است. برای استفاده نامحدود، حساب خود را به پریمیوم ارتقا دهید.");
            return;
        }

        $downloadCounter[$telegramId] = $current + 1;

        // ارسال مدیا بر اساس نوع ثبت‌شده در لینک
        $caption = $linkData['caption'] ?? null;
        switch ($linkData['type']) {
            case 'photo':
                $bot->sendPhoto($linkData['file_id'], caption: $caption);
                break;
            case 'video':
                $bot->sendVideo($linkData['file_id'], caption: $caption);
                break;
            case 'document':
            default:
                $bot->sendDocument($linkData['file_id'], caption: $caption);
                break;
        }

        // ثبت آنالیتیکس ساده
        try {
            $analyticsService->recordDownload($user->getId(), null);
            SimpleLogger::info('دانلود مدیا موفق', [
                'user_id' => $user->getId(),
                'telegram_id' => $telegramId,
                'media_type' => $linkData['type'],
            ]);
        } catch (Throwable $e) {
            SimpleLogger::error('خطا در ثبت آنالیتیکس دانلود', [
                'user_id' => $user->getId(),
                'error' => $e->getMessage(),
            ]);
        }
        
        return;
    }

    // شروع معمولی - نمایش منوی اصلی
    $user = $userService->getOrCreateByTelegramId($telegramId);
    $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
        || $adminService->getByTelegramId($telegramId) !== null;
    
    $modeText = $user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان';
    $welcomeText = "👋 سلام! به ربات آپلودر خوش آمدید." . PHP_EOL . PHP_EOL;
    $welcomeText .= "وضعیت حساب: {$modeText}";
    
    $bot->sendMessage(
        $welcomeText,
        reply_markup: buildMainMenu($isAdmin)
    );
});

// هندلر Callback Query برای navigation در منوها
$bot->onCallbackQuery(function (Nutgram $bot) use (
    $userService,
    $adminService,
    $mediaService,
    $analyticsService,
    $paymentService,
    $adsService,
    $ownerTelegramId,
    $nextpayApiKey,
    $zarinpalMerchantId,
    $zarinpalSandbox,
    $paymentCallbackBaseUrl,
) {
    $callbackQuery = $bot->callbackQuery();
    if ($callbackQuery === null) {
        return;
    }

    $data = $callbackQuery->data ?? '';
    $from = $callbackQuery->from;
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $message = $callbackQuery->message;
    $messageId = $message?->message_id ?? null;
    $chatId = $message?->chat->id ?? null;

    if ($messageId === null || $chatId === null) {
        return;
    }

    $user = $userService->getOrCreateByTelegramId($telegramId);
    $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
        || $adminService->getByTelegramId($telegramId) !== null;

    // Navigation در منوها
    if ($data === 'menu:main') {
        $modeText = $user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان';
        $text = "👋 منوی اصلی" . PHP_EOL . PHP_EOL . "وضعیت حساب: {$modeText}";
        $bot->editMessageText(
            $text,
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildMainMenu($isAdmin)
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin') {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '🔧 پنل مدیریت ادمین',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildAdminMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:premium') {
        if ($user->getMode() === 'premium') {
            $bot->answerCallbackQuery('شما قبلاً حساب پریمیوم دارید! 🎉', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '⭐ ارتقا به حساب پریمیوم' . PHP_EOL . PHP_EOL .
            'لطفاً روش پرداخت را انتخاب کنید:',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildPremiumMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin:media') {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }
        
        $admin = $adminService->getByTelegramId($telegramId);
        $allMedia = $mediaService->listByFolder(1); // در نسخه مینیمال folderId = 1
        
        // فیلتر کردن مدیاهای متعلق به این ادمین
        $userMedia = array_filter($allMedia, fn($m) => $m->getOwnerAdminId() === $admin->getId());
        $userMedia = array_values($userMedia);
        
        if (empty($userMedia)) {
            $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "هیچ مدیایی یافت نشد. برای افزودن مدیا، یک فایل را ارسال کنید.";
        } else {
            $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "لطفاً مدیای مورد نظر را انتخاب کنید:";
        }
        
        $bot->editMessageText(
            $text,
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildMediaListMenu($userMedia, 0)
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin:ads') {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '📢 مدیریت تبلیغات',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildAdsMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin:admins') {
        if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
            $bot->answerCallbackQuery('فقط مالک ربات می‌تواند ادمین‌ها را مدیریت کند!', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '👥 مدیریت ادمین‌ها',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildAdminsMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin:stats') {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '📊 آمار و گزارشات',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildAdminStatsMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:admin:broadcast') {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }
        
        $bot->editMessageText(
            '📤 برودکست' . PHP_EOL . PHP_EOL .
            'لطفاً نوع برودکست را انتخاب کنید:',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: buildBroadcastMenu()
        );
        $bot->answerCallbackQuery();
        return;
    }

    if ($data === 'menu:user:stats') {
        $downloads = $analyticsService->getDownloadCount(null, $user->getId());
        $text = "📊 آمار شخصی" . PHP_EOL . PHP_EOL;
        $text .= "📥 دانلودها: {$downloads}" . PHP_EOL;
        $text .= "وضعیت: " . ($user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان');
        
        $bot->editMessageText(
            $text,
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: InlineKeyboardMarkup::make()->addRow(
                InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
            )
        );
        $bot->answerCallbackQuery();
        return;
    }

    // Actions
    if (str_starts_with($data, 'action:premium:')) {
        $gateway = str_replace('action:premium:', '', $data);
        
        if ($user->getMode() === 'premium') {
            $bot->answerCallbackQuery('شما قبلاً حساب پریمیوم دارید! 🎉', show_alert: true);
            return;
        }

        if ($gateway === 'stars') {
            $premiumPrice = 100;
            $payload = 'premium_' . $telegramId . '_' . time();
            $payment = $paymentService->createPending(
                $user,
                'telegram_stars',
                $premiumPrice,
                $payload,
            );

            $bot->sendInvoice(
                title: 'ارتقا به حساب پریمیوم',
                description: 'با حساب پریمیوم، دانلود نامحدود داشته باشید و از تمام امکانات ربات استفاده کنید.',
                payload: $payload,
                provider_token: '',
                currency: 'XTR',
                prices: [
                    new LabeledPrice('حساب پریمیوم', $premiumPrice),
                ],
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($gateway === 'nextpay') {
            if (empty($nextpayApiKey)) {
                $bot->answerCallbackQuery('درگاه NextPay پیکربندی نشده است!', show_alert: true);
                return;
            }

            $premiumPrice = 500000;
            $orderId = 'premium_nextpay_' . $telegramId . '_' . time();
            $payment = $paymentService->createPending(
                $user,
                'nextpay',
                $premiumPrice,
                $orderId,
            );

            $nextpay = new NextPayGateway($nextpayApiKey);
            $callbackUrl = $paymentCallbackBaseUrl . '?gateway=nextpay';
            $result = $nextpay->createPayment($premiumPrice, $callbackUrl, $orderId);

            if (!$result['success']) {
                $bot->answerCallbackQuery('خطا در ایجاد لینک پرداخت!', show_alert: true);
                return;
            }

            $bot->sendMessage(
                "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
                $result['url'] . PHP_EOL . PHP_EOL .
                "مبلغ: " . number_format($premiumPrice) . " ریال"
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($gateway === 'zarinpal') {
            if (empty($zarinpalMerchantId)) {
                $bot->answerCallbackQuery('درگاه ZarinPal پیکربندی نشده است!', show_alert: true);
                return;
            }

            $premiumPrice = 500000;
            $authority = 'premium_zarinpal_' . $telegramId . '_' . time();
            $payment = $paymentService->createPending(
                $user,
                'zarinpal',
                $premiumPrice,
                $authority,
            );

            $zarinpal = new ZarinPalGateway($zarinpalMerchantId, $zarinpalSandbox);
            $callbackUrl = $paymentCallbackBaseUrl . '?gateway=zarinpal';
            $result = $zarinpal->createPayment($premiumPrice, $callbackUrl, 'ارتقا به حساب پریمیوم');

            if (!$result['success']) {
                $bot->answerCallbackQuery('خطا در ایجاد لینک پرداخت!', show_alert: true);
                return;
            }

            $realAuthority = $result['authority'] ?? '';
            if (!empty($realAuthority)) {
                $payment = $paymentService->findByPayload($authority);
                if ($payment !== null) {
                    $paymentService->setExternalId($payment, $realAuthority);
                }
            }

            $bot->sendMessage(
                "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
                $result['url'] . PHP_EOL . PHP_EOL .
                "مبلغ: " . number_format($premiumPrice) . " ریال"
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    // Actions برای ادمین‌ها
    if (str_starts_with($data, 'action:admin:')) {
        if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
            $bot->answerCallbackQuery('فقط مالک ربات می‌تواند ادمین‌ها را مدیریت کند!', show_alert: true);
            return;
        }

        $action = str_replace('action:admin:', '', $data);
        
        if ($action === 'list') {
            $admins = $adminService->getAll();
            if (empty($admins)) {
                $text = "هیچ ادمینی ثبت نشده است.";
            } else {
                $text = "👥 لیست ادمین‌ها:\n\n";
                foreach ($admins as $admin) {
                    $role = $admin->isOwner() ? "👑 Owner" : "👤 Admin";
                    $text .= "{$role}: {$admin->getTelegramId()}\n";
                }
            }
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildAdminsMenu()
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($action === 'add:input') {
            global $userStates;
            $userStates[$telegramId] = ['action' => 'add_admin', 'step' => 1];
            
            $bot->editMessageText(
                '➕ افزودن ادمین' . PHP_EOL . PHP_EOL .
                'لطفاً شناسه تلگرام ادمین جدید را ارسال کنید:' . PHP_EOL .
                '(مثال: 123456789)',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($action === 'remove:input') {
            global $userStates;
            $userStates[$telegramId] = ['action' => 'remove_admin', 'step' => 1];
            
            $bot->editMessageText(
                '🗑️ حذف ادمین' . PHP_EOL . PHP_EOL .
                'لطفاً شناسه تلگرام ادمینی که می‌خواهید حذف کنید را ارسال کنید:' . PHP_EOL .
                '(مثال: 123456789)',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    if (str_starts_with($data, 'action:ads:')) {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }

        $action = str_replace('action:ads:', '', $data);
        $admin = $adminService->getByTelegramId($telegramId);
        
        if ($action === 'list') {
            $ads = $adsService->getAllAdsForAdmin($admin->getId());
            if (empty($ads)) {
                $text = "هیچ تبلیغی ثبت نشده است.";
            } else {
                $text = "📢 لیست تبلیغات:\n\n";
                foreach ($ads as $ad) {
                    $status = $ad->isEnabled() ? "✅ فعال" : "❌ غیرفعال";
                    $text .= "#{$ad->getId()} - {$status}\n";
                    $text .= "متن: " . mb_substr($ad->getText(), 0, 50) . "...\n";
                    $text .= "فرکانس: هر {$ad->getFrequency()} پیام\n\n";
                }
            }
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildAdsMenu()
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($action === 'create:input') {
            global $userStates;
            $userStates[$telegramId] = ['action' => 'create_ad', 'step' => 1];
            
            $bot->editMessageText(
                '➕ افزودن تبلیغ' . PHP_EOL . PHP_EOL .
                'لطفاً متن تبلیغ را ارسال کنید:',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    // Actions برای مدیریت مدیا
    if (str_starts_with($data, 'action:media:')) {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }

        if ($data === 'action:media:list:page:0' || str_starts_with($data, 'action:media:list:page:')) {
            $page = 0;
            if (preg_match('/action:media:list:page:(\d+)/', $data, $matches)) {
                $page = (int) $matches[1];
            }
            
            $admin = $adminService->getByTelegramId($telegramId);
            $allMedia = $mediaService->listByFolder(1); // در نسخه مینیمال folderId = 1
            
            // فیلتر کردن مدیاهای متعلق به این ادمین
            $userMedia = array_filter($allMedia, fn($m) => $m->getOwnerAdminId() === $admin->getId());
            $userMedia = array_values($userMedia);
            
            if (empty($userMedia)) {
                $text = "هیچ مدیایی یافت نشد.";
            } else {
                $currentPage = $page + 1;
                $text = "📁 لیست مدیاها ({$currentPage}):\n\n";
            }
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildMediaListMenu($userMedia, $page)
            );
            $bot->answerCallbackQuery();
            return;
        }

        if (preg_match('/action:media:select:(\d+)/', $data, $matches)) {
            $mediaId = (int) $matches[1];
            $media = $mediaService->findById($mediaId);
            
            if ($media === null) {
                $bot->answerCallbackQuery('مدیا یافت نشد!', show_alert: true);
                return;
            }
            
            $downloads = $analyticsService->getDownloadCount($mediaId);
            $clicks = $analyticsService->getClickCount($mediaId);
            
            $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
            $text .= "نوع: {$media->getType()}\n";
            $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            if ($media->getAutoDeleteSeconds() !== null) {
                $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
            } else {
                $text .= "Auto-delete: ❌ غیرفعال\n";
            }
            $text .= "دانلود: {$downloads}\n";
            $text .= "کلیک: {$clicks}\n";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildMediaDetailMenu($mediaId)
            );
            $bot->answerCallbackQuery();
            return;
        }

        if (preg_match('/action:media:info:(\d+)/', $data, $matches)) {
            $mediaId = (int) $matches[1];
            $media = $mediaService->findById($mediaId);
            
            if ($media === null) {
                $bot->answerCallbackQuery('مدیا یافت نشد!', show_alert: true);
                return;
            }
            
            $downloads = $analyticsService->getDownloadCount($mediaId);
            $clicks = $analyticsService->getClickCount($mediaId);
            
            $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
            $text .= "نوع: {$media->getType()}\n";
            $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            if ($media->getAutoDeleteSeconds() !== null) {
                $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
            } else {
                $text .= "Auto-delete: ❌ غیرفعال\n";
            }
            $text .= "دانلود: {$downloads}\n";
            $text .= "کلیک: {$clicks}\n";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildMediaDetailMenu($mediaId)
            );
            $bot->answerCallbackQuery();
            return;
        }

        if (preg_match('/action:media:toggle_thumb:(\d+)/', $data, $matches)) {
            $mediaId = (int) $matches[1];
            $media = $mediaService->findById($mediaId);
            
            if ($media === null || $media->getOwnerAdminId() !== $adminService->getByTelegramId($telegramId)?->getId()) {
                $bot->answerCallbackQuery('مدیا یافت نشد یا شما دسترسی ندارید!', show_alert: true);
                return;
            }
            
            $media->toggleThumbnail(!$media->isThumbnailEnabled());
            $mediaService->updateMedia($media);
            
            $status = $media->isThumbnailEnabled() ? "فعال" : "غیرفعال";
            $bot->answerCallbackQuery("Thumbnail {$status} شد ✅");
            
            // به‌روزرسانی منو
            $downloads = $analyticsService->getDownloadCount($mediaId);
            $clicks = $analyticsService->getClickCount($mediaId);
            $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
            $text .= "نوع: {$media->getType()}\n";
            $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            if ($media->getAutoDeleteSeconds() !== null) {
                $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
            } else {
                $text .= "Auto-delete: ❌ غیرفعال\n";
            }
            $text .= "دانلود: {$downloads}\n";
            $text .= "کلیک: {$clicks}\n";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildMediaDetailMenu($mediaId)
            );
            return;
        }

        if (preg_match('/action:media:toggle_link:(\d+)/', $data, $matches)) {
            $mediaId = (int) $matches[1];
            $media = $mediaService->findById($mediaId);
            
            if ($media === null || $media->getOwnerAdminId() !== $adminService->getByTelegramId($telegramId)?->getId()) {
                $bot->answerCallbackQuery('مدیا یافت نشد یا شما دسترسی ندارید!', show_alert: true);
                return;
            }
            
            $media->toggleLinkInCaption(!$media->isLinkInCaptionEnabled());
            $mediaService->updateMedia($media);
            
            $status = $media->isLinkInCaptionEnabled() ? "فعال" : "غیرفعال";
            $bot->answerCallbackQuery("لینک در کپشن {$status} شد ✅");
            
            // به‌روزرسانی منو
            $downloads = $analyticsService->getDownloadCount($mediaId);
            $clicks = $analyticsService->getClickCount($mediaId);
            $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
            $text .= "نوع: {$media->getType()}\n";
            $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
            if ($media->getAutoDeleteSeconds() !== null) {
                $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
            } else {
                $text .= "Auto-delete: ❌ غیرفعال\n";
            }
            $text .= "دانلود: {$downloads}\n";
            $text .= "کلیک: {$clicks}\n";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildMediaDetailMenu($mediaId)
            );
            return;
        }

        if (preg_match('/action:media:autodelete:input:(\d+)/', $data, $matches)) {
            $mediaId = (int) $matches[1];
            global $userStates;
            $userStates[$telegramId] = ['action' => 'set_autodelete', 'step' => 1, 'data' => ['media_id' => $mediaId]];
            
            $bot->editMessageText(
                '⏱️ تنظیم Auto-Delete' . PHP_EOL . PHP_EOL .
                'لطفاً تعداد ثانیه‌ها را ارسال کنید:' . PHP_EOL .
                '(0 برای غیرفعال کردن)',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    // Actions برای برودکست
    if (str_starts_with($data, 'action:broadcast:')) {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }

        if ($data === 'action:broadcast:input:all') {
            global $userStates;
            $userStates[$telegramId] = ['action' => 'broadcast', 'step' => 1, 'data' => ['target' => 'all']];
            
            $bot->editMessageText(
                '📤 برودکست به همه کاربران' . PHP_EOL . PHP_EOL .
                'لطفاً شناسه مدیا را ارسال کنید:' . PHP_EOL .
                '(از منوی مدیریت مدیا می‌توانید شناسه را ببینید)',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($data === 'action:broadcast:input:channel') {
            global $userStates;
            $userStates[$telegramId] = ['action' => 'broadcast', 'step' => 1, 'data' => ['target' => 'channel']];
            
            $bot->editMessageText(
                '📤 برودکست به کانال/گروه' . PHP_EOL . PHP_EOL .
                'لطفاً شناسه مدیا را ارسال کنید:' . PHP_EOL .
                '(از منوی مدیریت مدیا می‌توانید شناسه را ببینید)',
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    // Action لغو
    if ($data === 'action:cancel') {
        global $userStates;
        unset($userStates[$telegramId]);
        
        $bot->editMessageText(
            '❌ عملیات لغو شد.',
            chat_id: $chatId,
            message_id: $messageId,
            reply_markup: InlineKeyboardMarkup::make()->addRow(
                InlineKeyboardButton::make('◀️ بازگشت به منوی اصلی', callback_data: 'menu:main')
            )
        );
        $bot->answerCallbackQuery();
        return;
    }

    if (str_starts_with($data, 'action:stats:')) {
        if (!$isAdmin) {
            $bot->answerCallbackQuery('شما دسترسی ادمین ندارید!', show_alert: true);
            return;
        }

        $action = str_replace('action:stats:', '', $data);
        
        if ($action === 'general') {
            $totalDownloads = $analyticsService->getDownloadCount();
            $totalClicks = $analyticsService->getClickCount();
            $joinConverted = $analyticsService->getJoinConvertedCount();
            $totalRevenue = $analyticsService->getTotalRevenue();
            $conversionRate = $analyticsService->getConversionRate();

            $text = "📊 آمار کلی ربات" . PHP_EOL . PHP_EOL .
                "📥 کل دانلودها: {$totalDownloads}" . PHP_EOL .
                "👆 کل کلیک‌ها: {$totalClicks}" . PHP_EOL .
                "✅ جوین‌های تبدیل شده: {$joinConverted}" . PHP_EOL .
                "💰 کل پرداخت‌ها: {$totalRevenue}" . PHP_EOL .
                "📈 نرخ تبدیل کلی: " . number_format($conversionRate, 2) . "%";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildAdminStatsMenu()
            );
            $bot->answerCallbackQuery();
            return;
        }

        if ($action === 'revenue') {
            $telegramStars = $analyticsService->getRevenueByGateway('telegram_stars');
            $nextpay = $analyticsService->getRevenueByGateway('nextpay');
            $zarinpal = $analyticsService->getRevenueByGateway('zarinpal');
            $total = $analyticsService->getTotalRevenue();

            $text = "💰 گزارش درآمد" . PHP_EOL . PHP_EOL .
                "⭐ Telegram Stars: {$telegramStars}" . PHP_EOL .
                "💳 NextPay: {$nextpay}" . PHP_EOL .
                "💳 ZarinPal: {$zarinpal}" . PHP_EOL .
                "📊 کل: {$total}";
            
            $bot->editMessageText(
                $text,
                chat_id: $chatId,
                message_id: $messageId,
                reply_markup: buildAdminStatsMenu()
            );
            $bot->answerCallbackQuery();
            return;
        }
    }

    $bot->answerCallbackQuery();
});

// هندلر پردازش ورودی‌های کاربر بر اساس state
$bot->onMessage(function (Nutgram $bot) use (
    $userService,
    $adminService,
    $mediaService,
    $adsService,
    $ownerTelegramId,
    &$userStates,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $message = $bot->message();
    if ($message === null) {
        return;
    }

    // بررسی state
    if (!isset($userStates[$telegramId])) {
        return; // state وجود ندارد، به handler بعدی برو
    }

    $state = $userStates[$telegramId];
    $text = $message->text ?? '';

    // اگر پیام مدیا است و state برای ورودی متنی است، نادیده بگیر
    if (!empty($message->photo) || $message->video !== null || $message->document !== null) {
        if (in_array($state['action'], ['add_admin', 'remove_admin', 'create_ad', 'set_autodelete', 'broadcast'], true)) {
            return; // به handler آپلود مدیا برو
        }
    }

    // پردازش state
    switch ($state['action']) {
        case 'add_admin':
            if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
                $bot->sendMessage("فقط مالک ربات می‌تواند ادمین اضافه کند.");
                unset($userStates[$telegramId]);
                return;
            }

            $newAdminId = (int) $text;
            if ($newAdminId === 0) {
                $bot->sendMessage("شناسه تلگرام معتبر نیست. لطفاً یک عدد معتبر ارسال کنید.");
                return;
            }

            $admin = $adminService->addAdmin($newAdminId);
            $bot->sendMessage("ادمین با شناسه {$newAdminId} اضافه شد. ✅");
            unset($userStates[$telegramId]);
            break;

        case 'remove_admin':
            if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
                $bot->sendMessage("فقط مالک ربات می‌تواند ادمین حذف کند.");
                unset($userStates[$telegramId]);
                return;
            }

            $adminIdToRemove = (int) $text;
            if ($adminIdToRemove === 0) {
                $bot->sendMessage("شناسه تلگرام معتبر نیست. لطفاً یک عدد معتبر ارسال کنید.");
                return;
            }

            $removed = $adminService->removeAdmin($adminIdToRemove);
            if ($removed) {
                $bot->sendMessage("ادمین با شناسه {$adminIdToRemove} حذف شد. ✅");
            } else {
                $bot->sendMessage("ادمین یافت نشد یا نمی‌توان owner را حذف کرد.");
            }
            unset($userStates[$telegramId]);
            break;

        case 'create_ad':
            $admin = $adminService->getByTelegramId($telegramId);
            if ($admin === null) {
                $bot->sendMessage("فقط ادمین‌ها می‌توانند تبلیغ ایجاد کنند.");
                unset($userStates[$telegramId]);
                return;
            }

            if (empty($text)) {
                $bot->sendMessage("متن تبلیغ نمی‌تواند خالی باشد.");
                return;
            }

            $ad = $adsService->createAd($admin->getId(), $text);
            $bot->sendMessage("تبلیغ با شناسه #{$ad->getId()} اضافه شد. ✅");
            unset($userStates[$telegramId]);
            break;

        case 'set_autodelete':
            $mediaId = $state['data']['media_id'] ?? 0;
            $media = $mediaService->findById($mediaId);
            
            if ($media === null) {
                $bot->sendMessage("مدیا یافت نشد.");
                unset($userStates[$telegramId]);
                return;
            }

            $admin = $adminService->getByTelegramId($telegramId);
            if ($admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                $bot->sendMessage("شما دسترسی به این مدیا ندارید.");
                unset($userStates[$telegramId]);
                return;
            }

            $seconds = (int) $text;
            $media->setAutoDeleteSeconds($seconds > 0 ? $seconds : null);
            $mediaService->updateMedia($media);

            if ($seconds > 0) {
                $bot->sendMessage("Auto-delete timer برای مدیا #{$mediaId} روی {$seconds} ثانیه تنظیم شد. ✅");
            } else {
                $bot->sendMessage("Auto-delete timer برای مدیا #{$mediaId} غیرفعال شد. ✅");
            }
            unset($userStates[$telegramId]);
            break;

        case 'broadcast':
            $target = $state['data']['target'] ?? 'all';
            
            if ($state['step'] === 1) {
                // دریافت media_id
                $mediaId = (int) $text;
                if ($mediaId === 0) {
                    $bot->sendMessage("شناسه مدیا معتبر نیست. لطفاً یک عدد معتبر ارسال کنید.");
                    return;
                }

                $media = $mediaService->findById($mediaId);
                if ($media === null) {
                    $bot->sendMessage("مدیا با شناسه {$mediaId} یافت نشد.");
                    unset($userStates[$telegramId]);
                    return;
                }

                if ($target === 'all') {
                    // ارسال به همه کاربران
                    $users = $userService->getAll();
                    $bot->sendMessage("شروع ارسال به " . count($users) . " کاربر...");
                    
                    $sentCount = 0;
                    $failedCount = 0;
                    
                    foreach ($users as $user) {
                        try {
                            $caption = $media->getCaption();
                            switch ($media->getType()) {
                                case 'photo':
                                    $bot->sendPhoto($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                                    break;
                                case 'video':
                                    $bot->sendVideo($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                                    break;
                                case 'document':
                                default:
                                    $bot->sendDocument($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                                    break;
                            }
                            $sentCount++;
                            usleep(50000); // 50ms
                        } catch (Throwable $e) {
                            $failedCount++;
                        }
                    }
                    
                    $bot->sendMessage(
                        "ارسال انجام شد! ✅" . PHP_EOL .
                        "موفق: {$sentCount}" . PHP_EOL .
                        "ناموفق: {$failedCount}"
                    );
                } else {
                    // نیاز به دریافت chat_id
                    $userStates[$telegramId] = ['action' => 'broadcast', 'step' => 2, 'data' => ['target' => $target, 'media_id' => $mediaId]];
                    $bot->sendMessage("لطفاً شناسه کانال/گروه را ارسال کنید:" . PHP_EOL . "(مثال: -1001234567890)");
                    return;
                }
            } elseif ($state['step'] === 2) {
                // دریافت chat_id
                $chatId = $text;
                $mediaId = $state['data']['media_id'] ?? 0;
                $media = $mediaService->findById($mediaId);
                
                if ($media === null) {
                    $bot->sendMessage("مدیا یافت نشد.");
                    unset($userStates[$telegramId]);
                    return;
                }

                try {
                    $caption = $media->getCaption();
                    switch ($media->getType()) {
                        case 'photo':
                            $bot->sendPhoto($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                            break;
                        case 'video':
                            $bot->sendVideo($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                            break;
                        case 'document':
                        default:
                            $bot->sendDocument($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                            break;
                    }
                    $bot->sendMessage("ارسال به {$chatId} انجام شد! ✅");
                } catch (Throwable $e) {
                    $bot->sendMessage("خطا در ارسال به {$chatId}: " . $e->getMessage());
                }
            }
            
            unset($userStates[$telegramId]);
            break;
    }
});

// هندلر آپلود مدیا توسط ادمین (نسخه مینیمال: photo / video / document)
$bot->onMessage(function (Nutgram $bot) use (
    $adminService,
    $mediaService,
    $linkStore,
    $ownerTelegramId,
    &$userStates,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // اگر کاربر در state است و state برای ورودی متنی است، نادیده بگیر
    if (isset($userStates[$telegramId]) && in_array($userStates[$telegramId]['action'], ['add_admin', 'remove_admin', 'create_ad', 'set_autodelete', 'broadcast'], true)) {
        return;
    }

    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        // فقط صاحب ربات اجازه آپلود دارد (نسخه‌ی مینیمال)
        return;
    }

    $admin = $adminService->getOrCreateOwner($telegramId);

    $message = $bot->message();
    if ($message === null) {
        return;
    }

    $type   = null;
    $fileId = null;
    $caption = $message->caption ?? null;

    if (!empty($message->photo)) {
        $photo  = end($message->photo);
        $type   = 'photo';
        $fileId = $photo?->file_id;
    } elseif ($message->video !== null) {
        $type   = 'video';
        $fileId = $message->video->file_id;
    } elseif ($message->document !== null) {
        $type   = 'document';
        $fileId = $message->document->file_id;
    }

    if ($fileId === null || $type === null) {
        // پیامی که مدیا نیست را نادیده بگیر
        return;
    }

    // در نسخه مینیمال، همه چیز در یک پوشه پیش‌فرض با شناسه 1 ذخیره می‌شود.
    $folderId = 1;

    $media = $mediaService->createMedia(
        folderId: $folderId,
        ownerAdminId: $admin->getId(),
        telegramFileId: $fileId,
        type: $type,
        caption: $caption,
    );

    $code = $linkStore->createForMedia($media);

    $bot->sendMessage(
        "مدیا ذخیره شد ✅" . PHP_EOL .
        "کد لینک یکتا: <code>{$code}</code>" . PHP_EOL .
        "لینک دیپ‌لینک: /start {$code}",
        parse_mode: 'HTML',
    );
});

// دستور /premium - هدایت به منوی پرداخت
$bot->onCommand('premium', function (Nutgram $bot) use (
    $userService,
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $user = $userService->getOrCreateByTelegramId($telegramId);
    $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
        || $adminService->getByTelegramId($telegramId) !== null;

    if ($user->getMode() === 'premium') {
        $bot->sendMessage("شما قبلاً حساب پریمیوم دارید! 🎉");
        return;
    }

    $bot->sendMessage(
        '⭐ ارتقا به حساب پریمیوم' . PHP_EOL . PHP_EOL .
        'لطفاً روش پرداخت را انتخاب کنید:',
        reply_markup: buildPremiumMenu()
    );
});

// هندلر PreCheckoutQuery برای بررسی قبل از پرداخت
$bot->onPreCheckoutQuery(function (Nutgram $bot) use ($paymentService) {
    $preCheckoutQuery = $bot->preCheckoutQuery();
    if ($preCheckoutQuery === null) {
        return;
    }

    $payload = $preCheckoutQuery->invoice_payload ?? '';
    $payment = $paymentService->findByPayload($payload);

    // اگر payment پیدا نشد یا قبلاً پرداخت شده است
    if ($payment === null) {
        $bot->answerPreCheckoutQuery(
            ok: false,
            error_message: 'پرداخت یافت نشد. لطفاً دوباره تلاش کنید.',
        );
        return;
    }

    if ($payment->getStatus() !== 'pending') {
        $bot->answerPreCheckoutQuery(
            ok: false,
            error_message: 'این پرداخت قبلاً پردازش شده است.',
        );
        return;
    }

    // بررسی مبلغ
    $expectedAmount = $payment->getAmount();
    $actualAmount = $preCheckoutQuery->total_amount ?? 0;

    if ($actualAmount !== $expectedAmount) {
        $bot->answerPreCheckoutQuery(
            ok: false,
            error_message: 'مبلغ پرداخت با مبلغ مورد انتظار مطابقت ندارد.',
        );
        return;
    }

    // تأیید پرداخت
    $bot->answerPreCheckoutQuery(ok: true);
});

// هندلر SuccessfulPayment برای پرداخت موفق
$bot->onSuccessfulPayment(function (Nutgram $bot) use (
    $userService,
    $paymentService,
    $analyticsService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $successfulPayment = $bot->message()?->successful_payment;

    if ($successfulPayment === null) {
        return;
    }

    $payload = $successfulPayment->invoice_payload ?? '';
    $payment = $paymentService->findByPayload($payload);

    if ($payment === null) {
        $bot->sendMessage("خطا در پردازش پرداخت. لطفاً با پشتیبانی تماس بگیرید.");
        return;
    }

    // ثبت پرداخت موفق
    try {
        $externalId = $successfulPayment->telegram_payment_charge_id ?? '';
        $paymentService->markSuccess($payment, $externalId);

        // ارتقا کاربر به پریمیوم
        $user = $userService->getOrCreateByTelegramId($telegramId);
        $userService->upgradeToPremium($user);

        // ثبت آنالیتیکس
        $analyticsService->recordPayment($user->getId(), $payment->getId());

        SimpleLogger::info('پرداخت Telegram Stars موفق', [
            'user_id' => $user->getId(),
            'telegram_id' => $telegramId,
            'payment_id' => $payment->getId(),
            'external_id' => $externalId,
        ]);
    } catch (Throwable $e) {
        SimpleLogger::error('خطا در پردازش پرداخت Telegram Stars', [
            'telegram_id' => $telegramId,
            'error' => $e->getMessage(),
        ]);
        $bot->sendMessage("خطا در پردازش پرداخت. لطفاً با پشتیبانی تماس بگیرید.");
        return;
    }

    $bot->sendMessage(
        "پرداخت با موفقیت انجام شد! ✅" . PHP_EOL .
        "حساب شما به پریمیوم ارتقا یافت. 🎉" . PHP_EOL .
        "اکنون می‌توانید از دانلود نامحدود استفاده کنید."
    );
});

// دستور /premium_nextpay - هدایت به منوی پرداخت (حذف شده - از منو استفاده کنید)
// دستور /premium_nextpay برای پرداخت با NextPay (برای سازگاری با نسخه‌های قدیمی)
$bot->onCommand('premium_nextpay', function (Nutgram $bot) use (
    $userService,
    $paymentService,
    $nextpayApiKey,
    $paymentCallbackBaseUrl,
) {
    if (empty($nextpayApiKey)) {
        $bot->sendMessage("درگاه NextPay پیکربندی نشده است.");
        return;
    }

    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $user = $userService->getOrCreateByTelegramId($telegramId);

    if ($user->getMode() === 'premium') {
        $bot->sendMessage("شما قبلاً حساب پریمیوم دارید! 🎉");
        return;
    }

    // ایجاد پرداخت pending
    $premiumPrice = 500000; // 500,000 ریال = 50,000 تومان
    $orderId = 'premium_nextpay_' . $telegramId . '_' . time();
    $payment = $paymentService->createPending(
        $user,
        'nextpay',
        $premiumPrice,
        $orderId,
    );

    // ایجاد لینک پرداخت
    $nextpay = new NextPayGateway($nextpayApiKey);
    $callbackUrl = $paymentCallbackBaseUrl . '?gateway=nextpay';
    $result = $nextpay->createPayment($premiumPrice, $callbackUrl, $orderId);

    if (!$result['success']) {
        $bot->sendMessage("خطا در ایجاد لینک پرداخت: " . ($result['error'] ?? 'خطای نامشخص'));
        return;
    }

    $bot->sendMessage(
        "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
        $result['url'] . PHP_EOL . PHP_EOL .
        "مبلغ: " . number_format($premiumPrice) . " ریال"
    );
});

// دستور /premium_zarinpal - هدایت به منوی پرداخت (حذف شده - از منو استفاده کنید)
// دستور /premium_zarinpal برای پرداخت با ZarinPal (برای سازگاری با نسخه‌های قدیمی)
$bot->onCommand('premium_zarinpal', function (Nutgram $bot) use (
    $userService,
    $paymentService,
    $zarinpalMerchantId,
    $zarinpalSandbox,
    $paymentCallbackBaseUrl,
) {
    if (empty($zarinpalMerchantId)) {
        $bot->sendMessage("درگاه ZarinPal پیکربندی نشده است.");
        return;
    }

    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $user = $userService->getOrCreateByTelegramId($telegramId);

    if ($user->getMode() === 'premium') {
        $bot->sendMessage("شما قبلاً حساب پریمیوم دارید! 🎉");
        return;
    }

    // ایجاد پرداخت pending
    $premiumPrice = 500000; // 500,000 ریال = 50,000 تومان
    $authority = 'premium_zarinpal_' . $telegramId . '_' . time();
    $payment = $paymentService->createPending(
        $user,
        'zarinpal',
        $premiumPrice,
        $authority,
    );

    // ایجاد لینک پرداخت
    $zarinpal = new ZarinPalGateway($zarinpalMerchantId, $zarinpalSandbox);
    $callbackUrl = $paymentCallbackBaseUrl . '?gateway=zarinpal';
    $result = $zarinpal->createPayment($premiumPrice, $callbackUrl, 'ارتقا به حساب پریمیوم');

    if (!$result['success']) {
        $bot->sendMessage("خطا در ایجاد لینک پرداخت: " . ($result['error'] ?? 'خطای نامشخص'));
        return;
    }

    // ذخیره authority واقعی در external_id برای استفاده در callback
    $realAuthority = $result['authority'] ?? '';
    if (!empty($realAuthority)) {
        $payment = $paymentService->findByPayload($authority);
        if ($payment !== null) {
            // ذخیره authority در external_id بدون تغییر status
            $paymentService->setExternalId($payment, $realAuthority);
        }
    }

    $bot->sendMessage(
        "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
        $result['url'] . PHP_EOL . PHP_EOL .
        "مبلغ: " . number_format($premiumPrice) . " ریال"
    );
});

// دستور /broadcast برای ارسال/فوروارد گروهی مدیا
// استفاده: /broadcast <media_id> [all|users|channel]
$bot->onCommand('broadcast', function (Nutgram $bot) use (
    $adminService,
    $mediaService,
    $userService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // فقط ادمین‌ها می‌توانند broadcast کنند
    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 3);

    if (count($parts) < 2) {
        $bot->sendMessage(
            "استفاده: /broadcast <media_id> [all|users|channel_id]" . PHP_EOL .
            "مثال: /broadcast 1 all" . PHP_EOL .
            "مثال: /broadcast 1 -1001234567890"
        );
        return;
    }

    $mediaId = (int) ($parts[1] ?? 0);
    $target = $parts[2] ?? 'all';

    $media = $mediaService->findById($mediaId);
    if ($media === null) {
        $bot->sendMessage("مدیا با شناسه {$mediaId} یافت نشد.");
        return;
    }

    $sentCount = 0;
    $failedCount = 0;

    if ($target === 'all' || $target === 'users') {
        // ارسال به همه کاربران
        $users = $userService->getAll();
        $bot->sendMessage("شروع ارسال به " . count($users) . " کاربر...");

        foreach ($users as $user) {
            try {
                $caption = $media->getCaption();
                switch ($media->getType()) {
                    case 'photo':
                        $bot->sendPhoto($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                        break;
                    case 'video':
                        $bot->sendVideo($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                        break;
                    case 'document':
                    default:
                        $bot->sendDocument($media->getTelegramFileId(), chat_id: $user->getTelegramId(), caption: $caption);
                        break;
                }
                $sentCount++;
                // تأخیر برای رعایت محدودیت‌های API تلگرام
                usleep(50000); // 50ms
            } catch (Throwable $e) {
                $failedCount++;
            }
        }
    } else {
        // ارسال به کانال/گروه خاص
        $chatId = $target;
        try {
            $caption = $media->getCaption();
            switch ($media->getType()) {
                case 'photo':
                    $bot->sendPhoto($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                    break;
                case 'video':
                    $bot->sendVideo($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                    break;
                case 'document':
                default:
                    $bot->sendDocument($media->getTelegramFileId(), chat_id: $chatId, caption: $caption);
                    break;
            }
            $sentCount = 1;
        } catch (Throwable $e) {
            $failedCount = 1;
            $bot->sendMessage("خطا در ارسال به {$chatId}: " . $e->getMessage());
            return;
        }
    }

    $bot->sendMessage(
        "ارسال انجام شد! ✅" . PHP_EOL .
        "موفق: {$sentCount}" . PHP_EOL .
        "ناموفق: {$failedCount}"
    );
});

// دستور /forward برای فوروارد گروهی مدیا
// استفاده: /forward <message_id> <target_chat_id>
$bot->onCommand('forward', function (Nutgram $bot) use (
    $adminService,
    $userService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // فقط ادمین‌ها می‌توانند forward کنند
    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 3);

    if (count($parts) < 3) {
        $bot->sendMessage(
            "استفاده: /forward <message_id> <target_chat_id>" . PHP_EOL .
            "مثال: /forward 123 -1001234567890" . PHP_EOL .
            "یا: /forward 123 all (برای فوروارد به همه کاربران)"
        );
        return;
    }

    $messageId = (int) ($parts[1] ?? 0);
    $target = $parts[2] ?? '';

    $fromChatId = $bot->chatId();
    if ($fromChatId === null) {
        $bot->sendMessage("خطا: شناسه چت یافت نشد.");
        return;
    }

    $sentCount = 0;
    $failedCount = 0;

    if ($target === 'all') {
        // فوروارد به همه کاربران
        $users = $userService->getAll();
        $bot->sendMessage("شروع فوروارد به " . count($users) . " کاربر...");

        foreach ($users as $user) {
            try {
                $bot->forwardMessage(
                    chat_id: $user->getTelegramId(),
                    from_chat_id: $fromChatId,
                    message_id: $messageId,
                );
                $sentCount++;
                // تأخیر برای رعایت محدودیت‌های API تلگرام
                usleep(50000); // 50ms
            } catch (Throwable $e) {
                $failedCount++;
            }
        }
    } else {
        // فوروارد به کانال/گروه خاص
        try {
            $bot->forwardMessage(
                chat_id: $target,
                from_chat_id: $fromChatId,
                message_id: $messageId,
            );
            $sentCount = 1;
        } catch (Throwable $e) {
            $failedCount = 1;
            $bot->sendMessage("خطا در فوروارد به {$target}: " . $e->getMessage());
            return;
        }
    }

    $bot->sendMessage(
        "فوروارد انجام شد! ✅" . PHP_EOL .
        "موفق: {$sentCount}" . PHP_EOL .
        "ناموفق: {$failedCount}"
    );
});

// دستور /stats - هدایت به منوی آمار
$bot->onCommand('stats', function (Nutgram $bot) use (
    $userService,
    $adminService,
    $analyticsService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
        || $adminService->getByTelegramId($telegramId) !== null;

    if ($isAdmin) {
        $bot->sendMessage(
            '📊 آمار و گزارشات',
            reply_markup: buildAdminStatsMenu()
        );
    } else {
        $user = $userService->getOrCreateByTelegramId($telegramId);
        $downloads = $analyticsService->getDownloadCount(null, $user->getId());
        $text = "📊 آمار شخصی" . PHP_EOL . PHP_EOL;
        $text .= "📥 دانلودها: {$downloads}" . PHP_EOL;
        $text .= "وضعیت: " . ($user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان');
        
        $bot->sendMessage($text);
    }
});

// دستور /revenue - هدایت به منوی آمار (حذف شده - از منو استفاده کنید)

// دستور /admins - هدایت به منوی مدیریت ادمین‌ها
$bot->onCommand('admins', function (Nutgram $bot) use (
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // فقط owner می‌تواند لیست ادمین‌ها را ببیند
    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        $bot->sendMessage("فقط مالک ربات می‌تواند از این دستور استفاده کند.");
        return;
    }

    $bot->sendMessage(
        '👥 مدیریت ادمین‌ها',
        reply_markup: buildAdminsMenu()
    );
});

// دستور /addadmin برای افزودن ادمین (نیاز به ورودی دارد - نگه داشته می‌شود)
$bot->onCommand('addadmin', function (Nutgram $bot) use (
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // فقط owner می‌تواند ادمین اضافه کند
    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        $bot->sendMessage("فقط مالک ربات می‌تواند از این دستور استفاده کند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage(
            "استفاده: /addadmin <telegram_id>" . PHP_EOL . PHP_EOL .
            "یا از منوی مدیریت ادمین‌ها استفاده کنید:",
            reply_markup: buildAdminsMenu()
        );
        return;
    }

    $newAdminId = (int) ($parts[1] ?? 0);
    if ($newAdminId === 0) {
        $bot->sendMessage("شناسه تلگرام معتبر نیست.");
        return;
    }

    $admin = $adminService->addAdmin($newAdminId);
    $bot->sendMessage("ادمین با شناسه {$newAdminId} اضافه شد. ✅");
});

// دستور /removeadmin برای حذف ادمین
$bot->onCommand('removeadmin', function (Nutgram $bot) use (
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;

    // فقط owner می‌تواند ادمین حذف کند
    if ($ownerTelegramId === null || $telegramId !== $ownerTelegramId) {
        $bot->sendMessage("فقط مالک ربات می‌تواند از این دستور استفاده کند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /removeadmin <telegram_id>");
        return;
    }

    $adminIdToRemove = (int) ($parts[1] ?? 0);
    if ($adminIdToRemove === 0) {
        $bot->sendMessage("شناسه تلگرام معتبر نیست.");
        return;
    }

    $removed = $adminService->removeAdmin($adminIdToRemove);
    if ($removed) {
        $bot->sendMessage("ادمین با شناسه {$adminIdToRemove} حذف شد. ✅");
    } else {
        $bot->sendMessage("ادمین یافت نشد یا نمی‌توان owner را حذف کرد.");
    }
});

// دستور /ads - هدایت به منوی مدیریت تبلیغات
$bot->onCommand('ads', function (Nutgram $bot) use (
    $adminService,
    $ownerTelegramId,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
        || $adminService->getByTelegramId($telegramId) !== null;

    if (!$isAdmin) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $bot->sendMessage(
        '📢 مدیریت تبلیغات',
        reply_markup: buildAdsMenu()
    );
});

// دستور /addad برای افزودن تبلیغ
$bot->onCommand('addad', function (Nutgram $bot) use (
    $adminService,
    $adsService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /addad <متن تبلیغ>");
        return;
    }

    $adText = $parts[1] ?? '';
    $ad = $adsService->createAd($admin->getId(), $adText);

    $bot->sendMessage("تبلیغ با شناسه #{$ad->getId()} اضافه شد. ✅");
});

// دستور /togglead برای فعال/غیرفعال کردن تبلیغ
$bot->onCommand('togglead', function (Nutgram $bot) use (
    $adminService,
    $adsService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /togglead <ad_id>");
        return;
    }

    $adId = (int) ($parts[1] ?? 0);
    $ad = $adsService->findById($adId);

    if ($ad === null || $ad->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("تبلیغ یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $ad->toggle(!$ad->isEnabled());
    $adsService->updateAd($ad);

    $status = $ad->isEnabled() ? "فعال" : "غیرفعال";
    $bot->sendMessage("تبلیغ #{$adId} {$status} شد. ✅");
});

// دستور /deletead برای حذف تبلیغ
$bot->onCommand('deletead', function (Nutgram $bot) use (
    $adminService,
    $adsService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /deletead <ad_id>");
        return;
    }

    $adId = (int) ($parts[1] ?? 0);
    $ad = $adsService->findById($adId);

    if ($ad === null || $ad->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("تبلیغ یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $adsService->deleteAd($adId);
    $bot->sendMessage("تبلیغ #{$adId} حذف شد. ✅");
});

// دستور /mediainfo برای نمایش اطلاعات مدیا
$bot->onCommand('mediainfo', function (Nutgram $bot) use (
    $adminService,
    $mediaService,
    $analyticsService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /mediainfo <media_id>");
        return;
    }

    $mediaId = (int) ($parts[1] ?? 0);
    $media = $mediaService->findById($mediaId);

    if ($media === null || $media->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("مدیا یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $downloads = $analyticsService->getDownloadCount($mediaId);
    $clicks = $analyticsService->getClickCount($mediaId);

    $info = "📊 اطلاعات مدیا #{$mediaId}\n\n";
    $info .= "نوع: {$media->getType()}\n";
    $info .= "پوشه: #{$media->getFolderId()}\n";
    $info .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
    $info .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
    if ($media->getAutoDeleteSeconds() !== null) {
        $info .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
    } else {
        $info .= "Auto-delete: ❌ غیرفعال\n";
    }
    $info .= "دانلود: {$downloads}\n";
    $info .= "کلیک: {$clicks}\n";
    if ($media->getCaption() !== null) {
        $info .= "کپشن: " . mb_substr($media->getCaption(), 0, 100) . "...\n";
    }

    $bot->sendMessage($info);
});

// دستور /toggletumbnail برای فعال/غیرفعال کردن Thumbnail
$bot->onCommand('toggletumbnail', function (Nutgram $bot) use (
    $adminService,
    $mediaService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /toggletumbnail <media_id>");
        return;
    }

    $mediaId = (int) ($parts[1] ?? 0);
    $media = $mediaService->findById($mediaId);

    if ($media === null || $media->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("مدیا یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $media->toggleThumbnail(!$media->isThumbnailEnabled());
    $mediaService->updateMedia($media);

    $status = $media->isThumbnailEnabled() ? "فعال" : "غیرفعال";
    $bot->sendMessage("Thumbnail مدیا #{$mediaId} {$status} شد. ✅");
});

// دستور /togglelinkcaption برای فعال/غیرفعال کردن لینک در کپشن
$bot->onCommand('togglelinkcaption', function (Nutgram $bot) use (
    $adminService,
    $mediaService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /togglelinkcaption <media_id>");
        return;
    }

    $mediaId = (int) ($parts[1] ?? 0);
    $media = $mediaService->findById($mediaId);

    if ($media === null || $media->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("مدیا یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $media->toggleLinkInCaption(!$media->isLinkInCaptionEnabled());
    $mediaService->updateMedia($media);

    $status = $media->isLinkInCaptionEnabled() ? "فعال" : "غیرفعال";
    $bot->sendMessage("لینک در کپشن مدیا #{$mediaId} {$status} شد. ✅");
});

// دستور /setautodelete برای تنظیم Auto-delete timer
$bot->onCommand('setautodelete', function (Nutgram $bot) use (
    $adminService,
    $mediaService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 3);

    if (count($parts) < 3) {
        $bot->sendMessage("استفاده: /setautodelete <media_id> <seconds> (0 برای غیرفعال کردن)");
        return;
    }

    $mediaId = (int) ($parts[1] ?? 0);
    $seconds = (int) ($parts[2] ?? 0);
    $media = $mediaService->findById($mediaId);

    if ($media === null || $media->getOwnerAdminId() !== $admin->getId()) {
        $bot->sendMessage("مدیا یافت نشد یا شما دسترسی ندارید.");
        return;
    }

    $media->setAutoDeleteSeconds($seconds > 0 ? $seconds : null);
    $mediaService->updateMedia($media);

    if ($seconds > 0) {
        $bot->sendMessage("Auto-delete timer برای مدیا #{$mediaId} روی {$seconds} ثانیه تنظیم شد. ✅");
    } else {
        $bot->sendMessage("Auto-delete timer برای مدیا #{$mediaId} غیرفعال شد. ✅");
    }
});

// ذخیره‌سازی پیام‌های Forced-Seen
/** @var array<string,bool> keyed by "chat_id:message_id" */
$forcedSeenMessages = [];

// هندلر برای Reactions
$bot->onMessageReaction(function (Nutgram $bot) use (&$forcedSeenMessages) {
    $messageReaction = $bot->messageReaction();
    if ($messageReaction === null) {
        return;
    }

    $chatId = $bot->chatId();
    $messageId = $bot->messageId();
    if ($chatId === null || $messageId === null) {
        return;
    }

    $key = "{$chatId}:{$messageId}";
    if (isset($forcedSeenMessages[$key])) {
        // کاربر reaction داده است، پیام دیده شده است
        $forcedSeenMessages[$key] = true;
    }
});

// دستور /forceseen برای ارسال پیام با Forced-Seen
$bot->onCommand('forceseen', function (Nutgram $bot) use (
    &$forcedSeenMessages,
    $adminService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 2);

    if (count($parts) < 2) {
        $bot->sendMessage("استفاده: /forceseen <متن پیام>");
        return;
    }

    $messageText = $parts[1] ?? '';
    $sentMessage = $bot->sendMessage($messageText);

    if ($sentMessage !== null) {
        $chatId = $sentMessage->chat->id;
        $messageId = $sentMessage->message_id;
        $key = "{$chatId}:{$messageId}";
        $forcedSeenMessages[$key] = false; // منتظر reaction هستیم

        // درخواست reaction
        try {
            $bot->setMessageReaction($chatId, $messageId, ['👍']);
        } catch (Throwable) {
            // اگر reaction پشتیبانی نشد، نادیده می‌گیریم
        }
    }
});

// دستور /checkforceseen برای بررسی وضعیت Forced-Seen
$bot->onCommand('checkforceseen', function (Nutgram $bot) use (
    &$forcedSeenMessages,
    $adminService,
) {
    $from = $bot->user();
    if ($from === null) {
        return;
    }

    $telegramId = (int) $from->id;
    $admin = $adminService->getByTelegramId($telegramId);

    if ($admin === null) {
        $bot->sendMessage("فقط ادمین‌ها می‌توانند از این دستور استفاده کنند.");
        return;
    }

    $message = $bot->message();
    $text = $message?->text ?? '';
    $parts = explode(' ', $text, 3);

    if (count($parts) < 3) {
        $bot->sendMessage("استفاده: /checkforceseen <chat_id> <message_id>");
        return;
    }

    $chatId = $parts[1] ?? '';
    $messageId = (int) ($parts[2] ?? 0);
    $key = "{$chatId}:{$messageId}";

    if (isset($forcedSeenMessages[$key])) {
        $status = $forcedSeenMessages[$key] ? "✅ دیده شده" : "⏳ منتظر reaction";
        $bot->sendMessage("وضعیت پیام: {$status}");
    } else {
        $bot->sendMessage("پیام یافت نشد یا Forced-Seen فعال نیست.");
    }
});

// اجرای ربات (برای Long Polling؛ در حالت وب‌هوک نیز از همین ورودی استفاده می‌شود)
$bot->run();


