<?php

declare(strict_types=1);

namespace Bot\Tests\Integration;

use Bot\Domain\Entity\Media;
use Bot\Domain\Entity\User;
use Bot\Domain\Repository\MediaRepositoryInterface;
use Bot\Domain\Repository\UserRepositoryInterface;
use Bot\Domain\Service\ForcedJoinService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\UserService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use Bot\Infrastructure\Repository\InMemoryMediaRepository;
use Bot\Infrastructure\Repository\InMemoryUserRepository;
use PHPUnit\Framework\TestCase;

/**
 * تست یکپارچه سبک برای سناریو جوین اجباری و تحویل مدیا
 */
final class ForcedJoinFlowTest extends TestCase
{
    private UserRepositoryInterface $userRepository;
    private MediaRepositoryInterface $mediaRepository;
    private UserService $userService;
    private MediaService $mediaService;
    private ForcedJoinService $forcedJoinService;
    private InMemoryLinkStore $linkStore;

    protected function setUp(): void
    {
        $this->userRepository = new InMemoryUserRepository();
        $this->mediaRepository = new InMemoryMediaRepository();
        $this->userService = new UserService($this->userRepository);
        $this->mediaService = new MediaService($this->mediaRepository);
        $this->forcedJoinService = new ForcedJoinService();
        $this->linkStore = new InMemoryLinkStore();
    }

    public function testUserRequestsMediaViaUniqueLinkWithoutJoin(): void
    {
        // ایجاد مدیا
        $media = $this->mediaService->createMedia(1, 10, 'file123', 'photo', 'Test media');
        $uniqueLink = $this->linkStore->createForMedia($media);

        // کاربر لینک را باز می‌کند
        $user = $this->userService->getOrCreateByTelegramId(12345);

        // بررسی وضعیت جوین کاربر (هنوز جوین نشده)
        $hasJoined = false;
        $isAllowed = $this->forcedJoinService->isUserAllowedToDownload($hasJoined);

        // کاربر نباید اجازه دانلود داشته باشد
        $this->assertFalse($isAllowed);
    }

    public function testUserCompletesForcedJoinAndReceivesMedia(): void
    {
        // ایجاد مدیا
        $media = $this->mediaService->createMedia(1, 10, 'file123', 'photo', 'Test media');
        $uniqueLink = $this->linkStore->createForMedia($media);

        // کاربر لینک را باز می‌کند
        $user = $this->userService->getOrCreateByTelegramId(12345);

        // کاربر جوین اجباری را کامل می‌کند
        $hasJoined = true;
        $isAllowed = $this->forcedJoinService->isUserAllowedToDownload($hasJoined);

        // کاربر باید اجازه دانلود داشته باشد
        $this->assertTrue($isAllowed);

        // بررسی اینکه لینک یکتا معتبر است
        $linkData = $this->linkStore->resolve($uniqueLink);
        $this->assertNotNull($linkData);
        $this->assertEquals($media->getTelegramFileId(), $linkData['file_id']);

        // مدیا باید قابل بازیابی باشد
        $mediaList = $this->mediaService->listByFolder(1);
        $this->assertNotEmpty($mediaList);
        $retrievedMedia = $mediaList[0];
        $this->assertEquals($media->getTelegramFileId(), $retrievedMedia->getTelegramFileId());
    }
}

