<?php

declare(strict_types=1);

namespace Bot\Tests\Integration;

use Bot\Domain\Entity\Admin;
use Bot\Domain\Repository\AdminRepositoryInterface;
use Bot\Domain\Repository\MediaRepositoryInterface;
use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\MediaService;
use Bot\Infrastructure\Repository\InMemoryAdminRepository;
use Bot\Infrastructure\Repository\InMemoryMediaRepository;
use PHPUnit\Framework\TestCase;

/**
 * تست یکپارچه سبک برای سناریو آپلود مدیا توسط ادمین
 */
final class AdminUploadMediaTest extends TestCase
{
    private AdminRepositoryInterface $adminRepository;
    private MediaRepositoryInterface $mediaRepository;
    private AdminService $adminService;
    private MediaService $mediaService;

    protected function setUp(): void
    {
        $this->adminRepository = new InMemoryAdminRepository();
        $this->mediaRepository = new InMemoryMediaRepository();
        $this->adminService = new AdminService($this->adminRepository);
        $this->mediaService = new MediaService($this->mediaRepository);
    }

    public function testAdminUploadsSingleMedia(): void
    {
        // ایجاد ادمین
        $admin = new Admin(1, 12345, true);
        $this->adminRepository->save($admin);

        // ادمین یک فایل تکی آپلود می‌کند
        $folderId = 1;
        $telegramFileId = 'photo_file_123';
        $type = 'photo';
        $caption = 'Test photo caption';

        $media = $this->mediaService->createMedia(
            $folderId,
            $admin->getId(),
            $telegramFileId,
            $type,
            $caption
        );

        // بررسی ذخیره شدن مدیا
        $this->assertNotNull($media);
        $this->assertEquals($admin->getId(), $media->getOwnerAdminId());
        $this->assertEquals($telegramFileId, $media->getTelegramFileId());
        $this->assertEquals($type, $media->getType());
        $this->assertEquals($caption, $media->getCaption());

        // بررسی بازیابی مدیا
        $mediaList = $this->mediaService->listByFolder($folderId);
        $this->assertNotEmpty($mediaList);
        $retrievedMedia = $mediaList[0];
        $this->assertEquals($telegramFileId, $retrievedMedia->getTelegramFileId());
    }

    public function testAdminUploadsMediaAlbum(): void
    {
        // ایجاد ادمین
        $admin = new Admin(1, 12345, true);
        $this->adminRepository->save($admin);

        // ادمین یک آلبوم مدیا (Media Group) آپلود می‌کند
        $folderId = 1;
        $mediaItems = [];

        $mediaItems[] = $this->mediaService->createMedia(
            $folderId,
            $admin->getId(),
            'photo_file_1',
            'photo',
            'Photo 1'
        );

        $mediaItems[] = $this->mediaService->createMedia(
            $folderId,
            $admin->getId(),
            'photo_file_2',
            'photo',
            'Photo 2'
        );

        $mediaItems[] = $this->mediaService->createMedia(
            $folderId,
            $admin->getId(),
            'photo_file_3',
            'photo',
            'Photo 3'
        );

        // بررسی ذخیره شدن تمام آیتم‌های آلبوم
        $this->assertCount(3, $mediaItems);

        // بررسی اینکه همه مدیاها در یک پوشه هستند
        $mediaInFolder = $this->mediaService->listByFolder($folderId);
        $this->assertCount(3, $mediaInFolder);

        // بررسی اینکه همه مدیاها متعلق به همان ادمین هستند
        foreach ($mediaInFolder as $media) {
            $this->assertEquals($admin->getId(), $media->getOwnerAdminId());
        }
    }
}

