<?php

declare(strict_types=1);

namespace Bot\Domain\ValueObject;

/**
 * Enum برای نوع مدیا
 * این Enum تضمین می‌کند که فقط نوع‌های معتبر استفاده می‌شوند
 */
enum MediaType: string
{
    case PHOTO = 'photo';
    case VIDEO = 'video';
    case DOCUMENT = 'document';
    case AUDIO = 'audio';
    case ANIMATION = 'animation';
    case VOICE = 'voice';
    case VIDEO_NOTE = 'video_note';
    case STICKER = 'sticker';

    /**
     * بررسی اینکه آیا نوع مدیا معتبر است یا نه
     * 
     * @param string $type نوع مدیا
     * @return bool true اگر معتبر باشد، false در غیر این صورت
     */
    public static function isValid(string $type): bool
    {
        return self::tryFrom($type) !== null;
    }

    /**
     * دریافت لیست تمام نوع‌های معتبر
     * 
     * @return array<string> لیست نوع‌های معتبر
     */
    public static function getValidTypes(): array
    {
        return array_column(self::cases(), 'value');
    }

    /**
     * دریافت نوع‌های قابل ارسال (که در switch case استفاده می‌شوند)
     * 
     * @return array<string> لیست نوع‌های قابل ارسال
     */
    public static function getSendableTypes(): array
    {
        return [
            self::PHOTO->value,
            self::VIDEO->value,
            self::DOCUMENT->value,
        ];
    }

    /**
     * بررسی اینکه آیا نوع مدیا قابل ارسال است یا نه
     * 
     * @param string $type نوع مدیا
     * @return bool true اگر قابل ارسال باشد، false در غیر این صورت
     */
    public static function isSendable(string $type): bool
    {
        return in_array($type, self::getSendableTypes(), true);
    }
}

