<?php

declare(strict_types=1);

use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardMarkup;

/**
 * ساخت منوی اصلی (برای کاربران و ادمین‌ها)
 */
function buildMainMenu(bool $isAdmin): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if ($isAdmin) {
        $keyboard->addRow(
            InlineKeyboardButton::make('🔧 پنل ادمین', callback_data: 'menu:admin')
        );
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⭐ ارتقا به پریمیوم', callback_data: 'menu:premium')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📊 آمار شخصی', callback_data: 'menu:user:stats'),
        InlineKeyboardButton::make('📝 لاگ‌های من', callback_data: 'menu:user:logs')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی ادمین
 */
function buildAdminMenu(bool $databaseInstalled = false): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    // همیشه دکمه وضعیت دیتابیس را نمایش بده
    $keyboard->addRow(
        InlineKeyboardButton::make('🗄️ وضعیت دیتابیس', callback_data: 'action:admin:database_status')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📁 مدیریت مدیا', callback_data: 'menu:admin:media'),
        InlineKeyboardButton::make('📢 مدیریت تبلیغات', callback_data: 'menu:admin:ads')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('👥 مدیریت ادمین‌ها', callback_data: 'menu:admin:admins'),
        InlineKeyboardButton::make('📊 آمار و گزارشات', callback_data: 'menu:admin:stats')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🔗 مدیریت کانال‌های اجباری', callback_data: 'menu:admin:forced_join')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 برودکست', callback_data: 'menu:admin:broadcast'),
        InlineKeyboardButton::make('📝 مدیریت لاگ‌ها', callback_data: 'menu:admin:logs')
    );

    $keyboard->addRow(
        InlineKeyboardButton::make('🗑️ پاکسازی کلی', callback_data: 'action:admin:clear_all:confirm:1')
    );

    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی پرداخت
 */
function buildPremiumMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⭐ Telegram Stars', callback_data: 'action:premium:stars')
    );
    
    // $keyboard->addRow(
    //     InlineKeyboardButton::make('💳 NextPay', callback_data: 'action:premium:nextpay')
    // );
    
    // $keyboard->addRow(
    //     InlineKeyboardButton::make('💳 ZarinPal', callback_data: 'action:premium:zarinpal')
    // );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت مدیا (با لیست مدیاها)
 */
function buildMediaListMenu(array $mediaList, int $page = 0, int $perPage = 5): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if (empty($mediaList)) {
        $keyboard->addRow(
            InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
        );
        return $keyboard;
    }
    
    $start = $page * $perPage;
    $end = min($start + $perPage, count($mediaList));
    $mediaPage = array_slice($mediaList, $start, $perPage);
    
    foreach ($mediaPage as $media) {
        $keyboard->addRow(
            InlineKeyboardButton::make(
                "📁 #{$media->getId()} - {$media->getType()}",
                callback_data: "action:media:select:{$media->getId()}"
            )
        );
    }
    
    // دکمه‌های navigation
    $navRow = [];
    if ($page > 0) {
        $navRow[] = InlineKeyboardButton::make('◀️ قبلی', callback_data: "action:media:list:page:" . ($page - 1));
    }
    if ($end < count($mediaList)) {
        $navRow[] = InlineKeyboardButton::make('بعدی ▶️', callback_data: "action:media:list:page:" . ($page + 1));
    }
    if (!empty($navRow)) {
        $keyboard->addRow(...$navRow);
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت یک مدیا خاص
 */
function buildMediaDetailMenu(int $mediaId): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('ℹ️ اطلاعات مدیا', callback_data: "action:media:info:{$mediaId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🖼️ Toggle Thumbnail', callback_data: "action:media:toggle_thumb:{$mediaId}"),
        InlineKeyboardButton::make('🔗 Toggle Link Caption', callback_data: "action:media:toggle_link:{$mediaId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('⏱️ تنظیم Auto-Delete', callback_data: "action:media:autodelete:input:{$mediaId}")
    );

    $keyboard->addRow(
        InlineKeyboardButton::make('🗑️ حذف مدیا', callback_data: "action:media:delete:confirm:{$mediaId}")
    );

    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت به لیست', callback_data: 'action:media:list:page:0')
    );

    return $keyboard;
}

/**
 * ساخت منوی مدیریت تبلیغات
 */
function buildAdsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('➕ افزودن تبلیغ', callback_data: 'action:ads:create:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لیست تبلیغات', callback_data: 'action:ads:list')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی لیست تبلیغات با دکمه‌های مدیریت
 */
function buildAdsListMenu(array $adsList, int $page = 0, int $perPage = 5): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if (empty($adsList)) {
        $keyboard->addRow(
            InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin:ads')
        );
        return $keyboard;
    }
    
    $start = $page * $perPage;
    $end = min($start + $perPage, count($adsList));
    $adsPage = array_slice($adsList, $start, $perPage);
    
    foreach ($adsPage as $ad) {
        $status = $ad->isEnabled() ? "✅" : "❌";
        $keyboard->addRow(
            InlineKeyboardButton::make(
                "{$status} تبلیغ #{$ad->getId()}",
                callback_data: "action:ads:select:{$ad->getId()}"
            )
        );
    }
    
    // دکمه‌های navigation
    $navRow = [];
    if ($page > 0) {
        $navRow[] = InlineKeyboardButton::make('◀️ قبلی', callback_data: "action:ads:list:page:" . ($page - 1));
    }
    if ($end < count($adsList)) {
        $navRow[] = InlineKeyboardButton::make('بعدی ▶️', callback_data: "action:ads:list:page:" . ($page + 1));
    }
    if (!empty($navRow)) {
        $keyboard->addRow(...$navRow);
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin:ads')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت یک تبلیغ خاص
 */
function buildAdDetailMenu(int $adId): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🔄 Toggle', callback_data: "action:ads:toggle:{$adId}"),
        InlineKeyboardButton::make('🗑️ حذف', callback_data: "action:ads:delete:{$adId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت به لیست', callback_data: 'action:ads:list:page:0')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت ادمین‌ها
 */
function buildAdminsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('➕ افزودن ادمین', callback_data: 'action:admin:add:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🗑️ حذف ادمین', callback_data: 'action:admin:remove:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لیست ادمین‌ها', callback_data: 'action:admin:list')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی آمار ادمین
 */
function buildAdminStatsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📊 آمار کلی', callback_data: 'action:stats:general')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('👥 آمار کاربران', callback_data: 'action:stats:users')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('💰 گزارش درآمد', callback_data: 'action:stats:revenue')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی آمار کاربران (برای ادمین)
 */
function buildAdminUsersStatsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 امروز', callback_data: 'action:stats:users:today'),
        InlineKeyboardButton::make('📅 دیروز', callback_data: 'action:stats:users:yesterday')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 این هفته', callback_data: 'action:stats:users:week'),
        InlineKeyboardButton::make('📅 این ماه', callback_data: 'action:stats:users:month')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 امسال', callback_data: 'action:stats:users:year'),
        InlineKeyboardButton::make('📅 کل', callback_data: 'action:stats:users:all')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin:stats')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی آمار شخصی کاربر (با فیلتر زمانی)
 */
function buildUserStatsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 امروز', callback_data: 'action:user:stats:today'),
        InlineKeyboardButton::make('📅 دیروز', callback_data: 'action:user:stats:yesterday')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 این هفته', callback_data: 'action:user:stats:week'),
        InlineKeyboardButton::make('📅 این ماه', callback_data: 'action:user:stats:month')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 امسال', callback_data: 'action:user:stats:year'),
        InlineKeyboardButton::make('📅 کل', callback_data: 'action:user:stats:all')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی برودکست
 */
function buildBroadcastMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 ارسال به همه کاربران', callback_data: 'action:broadcast:input:all')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📤 ارسال به کانال/گروه', callback_data: 'action:broadcast:input:channel')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت لاگ‌ها
 */
function buildLogsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لاگ‌های اخیر', callback_data: 'action:logs:recent')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🔴 خطاها', callback_data: 'action:logs:level:ERROR'),
        InlineKeyboardButton::make('⚠️ هشدارها', callback_data: 'action:logs:level:WARNING')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('ℹ️ اطلاعات', callback_data: 'action:logs:level:INFO'),
        InlineKeyboardButton::make('🐛 دیباگ', callback_data: 'action:logs:level:DEBUG')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📅 امروز', callback_data: 'action:logs:period:today'),
        InlineKeyboardButton::make('📅 دیروز', callback_data: 'action:logs:period:yesterday')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🗑️ پاکسازی لاگ‌های قدیمی', callback_data: 'action:logs:cleanup:confirm')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت کانال‌های اجباری
 */
function buildForcedJoinChannelsMenu(): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('➕ افزودن کانال', callback_data: 'action:forced_join:add:input')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('📋 لیست کانال‌ها', callback_data: 'action:forced_join:list')
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی لیست کانال‌های اجباری
 */
function buildForcedJoinChannelsListMenu(array $channels, int $page = 0, int $perPage = 5): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    if (empty($channels)) {
        $keyboard->addRow(
            InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin:forced_join')
        );
        return $keyboard;
    }
    
    $start = $page * $perPage;
    $end = min($start + $perPage, count($channels));
    $channelsPage = array_slice($channels, $start, $perPage);
    
    foreach ($channelsPage as $channel) {
        $status = $channel->isEnabled() ? "✅" : "❌";
        $keyboard->addRow(
            InlineKeyboardButton::make(
                "{$status} {$channel->getChannelUsername()}",
                callback_data: "action:forced_join:select:{$channel->getId()}"
            )
        );
    }
    
    // دکمه‌های navigation
    $navRow = [];
    if ($page > 0) {
        $navRow[] = InlineKeyboardButton::make('◀️ قبلی', callback_data: "action:forced_join:list:page:" . ($page - 1));
    }
    if ($end < count($channels)) {
        $navRow[] = InlineKeyboardButton::make('بعدی ▶️', callback_data: "action:forced_join:list:page:" . ($page + 1));
    }
    if (!empty($navRow)) {
        $keyboard->addRow(...$navRow);
    }
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:admin:forced_join')
    );
    
    return $keyboard;
}

/**
 * ساخت منوی مدیریت یک کانال اجباری خاص
 */
function buildForcedJoinChannelDetailMenu(int $channelId): InlineKeyboardMarkup
{
    $keyboard = InlineKeyboardMarkup::make();
    
    $keyboard->addRow(
        InlineKeyboardButton::make('🔄 فعال/غیرفعال', callback_data: "action:forced_join:toggle:{$channelId}"),
        InlineKeyboardButton::make('🗑️ حذف', callback_data: "action:forced_join:delete:{$channelId}")
    );
    
    $keyboard->addRow(
        InlineKeyboardButton::make('◀️ بازگشت به لیست', callback_data: 'action:forced_join:list:page:0')
    );
    
    return $keyboard;
}
