<?php

declare(strict_types=1);

use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\AnalyticsService;
use Bot\Domain\Service\ForcedJoinChannelService;
use Bot\Domain\Service\LogService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\PaymentService;
use Bot\Domain\Service\UserService;
use Bot\Domain\Service\AdsService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use Bot\Infrastructure\Payment\NextPayGateway;
use Bot\Infrastructure\Payment\ZarinPalGateway;
use SergiX44\Nutgram\Telegram\Types\Payment\LabeledPrice;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton;
use SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardMarkup;
use SergiX44\Nutgram\Nutgram;

/**
 * هندلر Callback Query برای navigation در منوها و actions
 */
function registerCallbackQueryHandler(
    Nutgram $bot,
    UserService $userService,
    AdminService $adminService,
    MediaService $mediaService,
    AnalyticsService $analyticsService,
    PaymentService $paymentService,
    AdsService $adsService,
    ?int $ownerTelegramId,
    string $nextpayApiKey,
    string $zarinpalMerchantId,
    bool $zarinpalSandbox,
    string $paymentCallbackBaseUrl,
    array &$userStates,
    InMemoryLinkStore $linkStore,
    LogService $logService,
    ForcedJoinChannelService $forcedJoinChannelService,
): void {
    $bot->onCallbackQuery(function (Nutgram $bot) use (
        $userService,
        $adminService,
        $mediaService,
        $analyticsService,
        $paymentService,
        $adsService,
        $ownerTelegramId,
        $nextpayApiKey,
        $zarinpalMerchantId,
        $zarinpalSandbox,
        $paymentCallbackBaseUrl,
        &$userStates,
        $linkStore,
        $logService,
        $forcedJoinChannelService,
    ) {
        $callbackQuery = $bot->callbackQuery();
        if ($callbackQuery === null) {
            return;
        }

        $data = $callbackQuery->data ?? '';
        $from = $callbackQuery->from;
        if ($from === null) {
            return;
        }

        $telegramId = (int) $from->id;
        $message = $callbackQuery->message;
        $messageId = $message?->message_id ?? null;
        $chatId = $message?->chat->id ?? null;

        if ($messageId === null || $chatId === null) {
            return;
        }

        // پاسخ سریع به callback query برای جلوگیری از timeout
        try {
            safeAnswerCallbackQuery($bot);
        } catch (\Throwable $e) {
            // اگر callback query قدیمی باشد، خطا را نادیده می‌گیریم
            if (!str_contains($e->getMessage(), 'query is too old')) {
                throw $e;
            }
        }

        $user = $userService->getOrCreateByTelegramId($telegramId);
        
        // اگر owner است، به صورت خودکار در دیتابیس ذخیره می‌شود
        if ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) {
            $adminService->getOrCreateOwner($telegramId);
        }
        
        $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
            || $adminService->getByTelegramId($telegramId) !== null;

        // Navigation در منوها
        if ($data === 'menu:main') {
            $modeText = $user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان';
            $text = "👋 منوی اصلی" . PHP_EOL . PHP_EOL . "وضعیت حساب: {$modeText}";
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildMainMenu($isAdmin)
            );
            return;
        }

        if ($data === 'menu:admin') {
            if (!$isAdmin) {
                try {
                    safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                } catch (\Throwable) {
                    // نادیده گرفتن خطا
                }
                return;
            }
            
            // بررسی وضعیت نصب دیتابیس
            $databaseConfigFile = __DIR__ . '/../../../.database_installed';
            $databaseInstalled = file_exists($databaseConfigFile);
            
            safeEditMessageText(
                $bot,
                '🔧 پنل مدیریت ادمین',
                $chatId,
                $messageId,
                buildAdminMenu($databaseInstalled)
            );
            return;
        }
        
        // Action برای بررسی وضعیت دیتابیس
        if ($data === 'action:admin:database_status') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند وضعیت دیتابیس را بررسی کند!', true);
                return;
            }
            
            // بررسی تنظیمات دیتابیس
            $dbHost = $_ENV['DB_HOST'] ?? 'localhost';
            $dbPort = isset($_ENV['DB_PORT']) ? (int) $_ENV['DB_PORT'] : 3306;
            $dbName = $_ENV['DB_DATABASE'] ?? $_ENV['DB_NAME'] ?? '';
            $dbUser = $_ENV['DB_USERNAME'] ?? $_ENV['DB_USER'] ?? '';
            $dbPass = $_ENV['DB_PASSWORD'] ?? '';
            
            $statusText = "🗄️ وضعیت دیتابیس" . PHP_EOL . PHP_EOL;
            
            if (empty($dbName) || empty($dbUser)) {
                $statusText .= "❌ تنظیمات دیتابیس در فایل .env وارد نشده است." . PHP_EOL;
                $statusText .= PHP_EOL . "لطفاً تنظیمات زیر را در فایل .env وارد کنید:" . PHP_EOL;
                $statusText .= "DB_HOST=127.0.0.1" . PHP_EOL;
                $statusText .= "DB_PORT=3306" . PHP_EOL;
                $statusText .= "DB_DATABASE=your_database" . PHP_EOL;
                $statusText .= "DB_USERNAME=your_username" . PHP_EOL;
                $statusText .= "DB_PASSWORD=your_password";
                
                safeEditMessageText(
                    $bot,
                    $statusText,
                    $chatId,
                    $messageId,
                    buildAdminMenu(false)
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
            
            // بررسی فایل flag
            $databaseConfigFile = __DIR__ . '/../../../.database_installed';
            $installedDate = file_exists($databaseConfigFile) ? file_get_contents($databaseConfigFile) : null;
            
            $statusText .= "📋 تنظیمات:" . PHP_EOL;
            $statusText .= "Host: " . str_repeat('*', min(strlen($dbHost), 10)) . PHP_EOL;
            $statusText .= "Database: " . str_repeat('*', min(strlen($dbName), 10)) . PHP_EOL;
            $statusText .= "User: " . str_repeat('*', min(strlen($dbUser), 10)) . PHP_EOL;
            $statusText .= PHP_EOL;
            
            // تست اتصال و بررسی جداول
            try {
                // بررسی وجود extension PDO
                if (!extension_loaded('pdo') || !extension_loaded('pdo_mysql')) {
                    throw new RuntimeException('Extension PDO یا PDO_MySQL نصب نشده است!');
                }
                
                $dsn = "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4";
                $pdo = new PDO($dsn, $dbUser, $dbPass, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]);
                
                $statusText .= "✅ اتصال به دیتابیس موفق بود!" . PHP_EOL . PHP_EOL;
                
                // بررسی جداول
                $requiredTables = [
                    'users',
                    'admins',
                    'folders',
                    'media',
                    'payments',
                    'ads',
                    'analytics_events',
                    'logs',
                ];
                
                $statusText .= "📊 بررسی جداول:" . PHP_EOL;
                $allTablesExist = true;
                
                foreach ($requiredTables as $table) {
                    // استفاده از prepared statement برای امنیت بیشتر
                    $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
                    $stmt->execute([$table]);
                    $exists = $stmt->rowCount() > 0;
                    
                    if ($exists) {
                        // شمارش رکوردها - استفاده از whitelist برای نام جدول
                        $allowedTables = ['users', 'admins', 'folders', 'media', 'payments', 'ads', 'analytics_events', 'logs'];
                        if (in_array($table, $allowedTables, true)) {
                            $countStmt = $pdo->prepare("SELECT COUNT(*) as count FROM `{$table}`");
                            $countStmt->execute();
                            $count = $countStmt->fetch()['count'];
                            $statusText .= "✅ {$table}: موجود ({$count} رکورد)" . PHP_EOL;
                        } else {
                            $statusText .= "✅ {$table}: موجود" . PHP_EOL;
                        }
                    } else {
                        $statusText .= "❌ {$table}: موجود نیست" . PHP_EOL;
                        $allTablesExist = false;
                    }
                }
                
                if ($allTablesExist) {
                    $statusText .= PHP_EOL . "✅ همه جداول به درستی نصب شده‌اند!" . PHP_EOL;
                    if ($installedDate) {
                        $statusText .= "📅 تاریخ نصب: {$installedDate}" . PHP_EOL;
                    }
                    $statusText .= PHP_EOL . "✅ وضعیت دیتابیس: OK";
                } else {
                    $statusText .= PHP_EOL . "⚠️ برخی جداول نصب نشده‌اند." . PHP_EOL;
                    $statusText .= "لطفاً از دستور /crdb برای نصب دیتابیس استفاده کنید.";
                }
                
            } catch (PDOException $e) {
                $statusText .= "❌ خطا در اتصال به دیتابیس!" . PHP_EOL;
                $statusText .= "خطا: " . $e->getMessage() . PHP_EOL . PHP_EOL;
                $statusText .= "لطفاً تنظیمات دیتابیس را بررسی کنید.";
            } catch (Throwable $e) {
                $statusText .= "❌ خطای غیرمنتظره!" . PHP_EOL;
                $statusText .= "خطا: " . $e->getMessage();
                
                SimpleLogger::error('خطا در بررسی وضعیت دیتابیس', [
                    'telegram_id' => $telegramId,
                    'error' => $e->getMessage(),
                ]);
            }
            
            safeEditMessageText(
                $bot,
                $statusText,
                $chatId,
                $messageId,
                buildAdminMenu(true)
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // Action برای اجرای migration های دیتابیس
        if ($data === 'action:admin:migrate') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند migration ها را اجرا کند!', true);
                return;
            }

            safeAnswerCallbackQuery($bot, 'در حال اجرای migration ها...', false);

            try {
                require_once __DIR__ . '/../helpers.php';
                require_once __DIR__ . '/../database-repositories.php';
                
                $pdo = createDatabaseConnection();
                if ($pdo === null) {
                    $text = "❌ اتصال به دیتابیس برقرار نشد!" . PHP_EOL . PHP_EOL .
                        "لطفاً تنظیمات دیتابیس را در فایل .env بررسی کنید.";
                    
                    safeEditMessageText(
                        $bot,
                        $text,
                        $chatId,
                        $messageId,
                        buildAdminMenu(false)
                    );
                    return;
                }

                $result = runAllMigrations($pdo);

                $text = "📊 نتیجه اجرای Migration ها:" . PHP_EOL . PHP_EOL;
                $text .= "📁 کل فایل‌ها: {$result['total']}" . PHP_EOL;
                $text .= "✅ اجرا شده: {$result['executed']}" . PHP_EOL;
                $text .= "❌ ناموفق: {$result['failed']}" . PHP_EOL . PHP_EOL;

                if (!empty($result['messages'])) {
                    $text .= "📝 جزئیات:" . PHP_EOL;
                    foreach ($result['messages'] as $msg) {
                        $text .= $msg . PHP_EOL;
                    }
                }

                if ($result['success']) {
                    $text .= PHP_EOL . "✅ همه migration ها با موفقیت اجرا شدند!";
                } else {
                    $text .= PHP_EOL . "⚠️ برخی migration ها ناموفق بودند. لطفاً لاگ‌ها را بررسی کنید.";
                }

                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminMenu(true)
                );
            } catch (Throwable $e) {
                $text = "❌ خطا در اجرای migration ها!" . PHP_EOL . PHP_EOL .
                    "خطا: " . $e->getMessage();
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminMenu(false)
                );
            }
            return;
        }
        
        // Action برای نصب دیتابیس - حذف شده (استفاده از دستور /crdb)

        if ($data === 'menu:premium') {
            if ($user->getMode() === 'premium') {
                try {
                    safeAnswerCallbackQuery($bot, 'شما قبلاً حساب پریمیوم دارید! 🎉', true);
                } catch (\Throwable) {
                    // نادیده گرفتن خطا
                }
                return;
            }
            
            safeEditMessageText(
                $bot,
                '⭐ ارتقا به حساب پریمیوم' . PHP_EOL . PHP_EOL .
                'لطفاً روش پرداخت را انتخاب کنید:',
                $chatId,
                $messageId,
                buildPremiumMenu()
            );
            return;
        }

        if ($data === 'menu:admin:media') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
            if ($admin === null) {
                safeAnswerCallbackQuery($bot, 'خطا در دریافت اطلاعات ادمین!', true);
                return;
            }
            
            $allMedia = $mediaService->listByFolder(1); // در نسخه مینیمال folderId = 1
            
            // فیلتر کردن مدیاهای متعلق به این ادمین
            $userMedia = array_filter($allMedia, fn($m) => $m->getOwnerAdminId() === $admin->getId());
            $userMedia = array_values($userMedia);
            
            if (empty($userMedia)) {
                $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "هیچ مدیایی یافت نشد. برای افزودن مدیا، یک فایل را ارسال کنید.";
            } else {
                $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "لطفاً مدیای مورد نظر را انتخاب کنید:";
            }
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildMediaListMenu($userMedia, 0)
            );
            return;
        }

        if ($data === 'menu:admin:ads') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            safeEditMessageText(
                $bot,
                '📢 مدیریت تبلیغات',
                $chatId,
                $messageId,
                buildAdsMenu()
            );
            return;
        }

        if ($data === 'menu:admin:admins') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند ادمین‌ها را مدیریت کند!', true);
                return;
            }
            
            safeEditMessageText(
                $bot,
                '👥 مدیریت ادمین‌ها',
                $chatId,
                $messageId,
                buildAdminsMenu()
            );
            return;
        }

        if ($data === 'menu:admin:stats') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            safeEditMessageText(
                $bot,
                '📊 آمار و گزارشات',
                $chatId,
                $messageId,
                buildAdminStatsMenu()
            );
            return;
        }

        if ($data === 'menu:admin:broadcast') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            safeEditMessageText(
                $bot,
                '📤 برودکست' . PHP_EOL . PHP_EOL .
                'لطفاً نوع برودکست را انتخاب کنید:',
                $chatId,
                $messageId,
                buildBroadcastMenu()
            );
            return;
        }

        if ($data === 'menu:user:stats') {
            // نمایش منوی انتخاب بازه زمانی
            $text = "📊 آمار شخصی" . PHP_EOL . PHP_EOL;
            $text .= "لطفاً بازه زمانی مورد نظر را انتخاب کنید:";
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildUserStatsMenu()
            );
            return;
        }

        // منوی لاگ‌های کاربر
        if ($data === 'menu:user:logs') {
            $logs = $logService->getLogs(telegramId: $telegramId, limit: 20);
            
            if (empty($logs)) {
                $text = "📝 لاگ‌های من" . PHP_EOL . PHP_EOL . "هیچ لاگی یافت نشد.";
            } else {
                $text = "📝 لاگ‌های من (20 مورد آخر):" . PHP_EOL . PHP_EOL;
                foreach ($logs as $log) {
                    $level = $log['level'] ?? 'UNKNOWN';
                    $message = mb_substr($log['message'] ?? '', 0, 60);
                    $date = date('Y-m-d H:i:s', strtotime($log['created_at'] ?? 'now'));
                    
                    // انتخاب emoji بر اساس سطح
                    $levelEmoji = match($level) {
                        'ERROR', 'CRITICAL' => '🔴',
                        'WARNING' => '⚠️',
                        'INFO' => 'ℹ️',
                        'DEBUG' => '🐛',
                        default => '📝',
                    };
                    
                    $text .= "{$levelEmoji} [{$date}]" . PHP_EOL;
                    $text .= "{$message}" . PHP_EOL . PHP_EOL;
                }
            }

            $keyboard = InlineKeyboardMarkup::make();
            $keyboard->addRow(
                InlineKeyboardButton::make('🔄 به‌روزرسانی', callback_data: 'menu:user:logs'),
                InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:main')
            );

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                $keyboard
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // Actions برای آمار کاربر با فیلتر زمانی
        if (str_starts_with($data, 'action:user:stats:')) {
            $period = str_replace('action:user:stats:', '', $data);
            
            $now = new \DateTimeImmutable('now', new \DateTimeZone('Asia/Tehran'));
            $startDate = null;
            $endDate = $now;
            $periodText = '';
            
            switch ($period) {
                case 'today':
                    $startDate = $now->setTime(0, 0, 0);
                    $periodText = 'امروز';
                    break;
                case 'yesterday':
                    $yesterday = $now->modify('-1 day');
                    $startDate = $yesterday->setTime(0, 0, 0);
                    $endDate = $yesterday->setTime(23, 59, 59);
                    $periodText = 'دیروز';
                    break;
                case 'week':
                    $startDate = $now->modify('monday this week')->setTime(0, 0, 0);
                    $periodText = 'این هفته';
                    break;
                case 'month':
                    $startDate = $now->modify('first day of this month')->setTime(0, 0, 0);
                    $periodText = 'این ماه';
                    break;
                case 'year':
                    $startDate = $now->modify('first day of january this year')->setTime(0, 0, 0);
                    $periodText = 'امسال';
                    break;
                case 'all':
                    $startDate = new \DateTimeImmutable('1970-01-01');
                    $periodText = 'کل';
                    break;
                default:
                    $startDate = new \DateTimeImmutable('1970-01-01');
                    $periodText = 'کل';
            }
            
            $downloads = $analyticsService->getDownloadCountForUserInRange($user->getId(), $startDate, $endDate);
            $clicks = $analyticsService->getClickCountForUserInRange($user->getId(), $startDate, $endDate);
            $payments = $analyticsService->getPaymentCountForUserInRange($user->getId(), $startDate, $endDate);
            
            $text = "📊 آمار شخصی ({$periodText})" . PHP_EOL . PHP_EOL;
            $text .= "👤 شناسه کاربر: {$user->getId()}" . PHP_EOL;
            $text .= "📱 شناسه تلگرام: {$user->getTelegramId()}" . PHP_EOL . PHP_EOL;
            $text .= "📥 تعداد دانلودها: {$downloads}" . PHP_EOL;
            $text .= "👆 تعداد کلیک‌ها: {$clicks}" . PHP_EOL;
            $text .= "💳 تعداد پرداخت‌ها: {$payments}" . PHP_EOL . PHP_EOL;
            $text .= "وضعیت حساب: " . ($user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان');
            
            if ($user->getMode() === 'free') {
                $text .= PHP_EOL . PHP_EOL . "💡 برای استفاده نامحدود، حساب خود را به پریمیوم ارتقا دهید!";
            }
            
            $keyboard = InlineKeyboardMarkup::make();
            if ($user->getMode() === 'free') {
                $keyboard->addRow(
                    InlineKeyboardButton::make('⭐ ارتقا به پریمیوم', callback_data: 'menu:premium')
                );
            }
            $keyboard->addRow(
                InlineKeyboardButton::make('🔄 به‌روزرسانی', callback_data: "action:user:stats:{$period}"),
                InlineKeyboardButton::make('◀️ بازگشت', callback_data: 'menu:user:stats')
            );
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                $keyboard
            );
            return;
        }

        // Actions برای پرداخت
        if (str_starts_with($data, 'action:premium:')) {
            $gateway = str_replace('action:premium:', '', $data);
            
            if ($user->getMode() === 'premium') {
                safeAnswerCallbackQuery($bot, 'شما قبلاً حساب پریمیوم دارید! 🎉', true);
                return;
            }

            if ($gateway === 'stars') {
                $premiumPrice = 100;
                $payload = 'premium_' . $telegramId . '_' . time();
                $payment = $paymentService->createPending(
                    $user,
                    'telegram_stars',
                    $premiumPrice,
                    $payload,
                );

                $bot->sendInvoice(
                    title: 'ارتقا به حساب پریمیوم',
                    description: 'با حساب پریمیوم، دانلود نامحدود داشته باشید و از تمام امکانات ربات استفاده کنید.',
                    payload: $payload,
                    provider_token: '',
                    currency: 'XTR',
                    prices: [
                        new LabeledPrice('حساب پریمیوم', $premiumPrice),
                    ],
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($gateway === 'nextpay') {
                if (empty($nextpayApiKey)) {
                    safeAnswerCallbackQuery($bot, 'درگاه NextPay پیکربندی نشده است!', true);
                    return;
                }

                $premiumPrice = 500000;
                $orderId = 'premium_nextpay_' . $telegramId . '_' . time();
                $payment = $paymentService->createPending(
                    $user,
                    'nextpay',
                    $premiumPrice,
                    $orderId,
                );

                $nextpay = new NextPayGateway($nextpayApiKey);
                $callbackUrl = $paymentCallbackBaseUrl . '?gateway=nextpay';
                $result = $nextpay->createPayment($premiumPrice, $callbackUrl, $orderId);

                if (!$result['success']) {
                    safeAnswerCallbackQuery($bot, 'خطا در ایجاد لینک پرداخت!', true);
                    return;
                }

                $bot->sendMessage(
                    "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
                    $result['url'] . PHP_EOL . PHP_EOL .
                    "مبلغ: " . number_format($premiumPrice) . " ریال"
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($gateway === 'zarinpal') {
                if (empty($zarinpalMerchantId)) {
                    safeAnswerCallbackQuery($bot, 'درگاه ZarinPal پیکربندی نشده است!', true);
                    return;
                }

                $premiumPrice = 500000;
                $authority = 'premium_zarinpal_' . $telegramId . '_' . time();
                $payment = $paymentService->createPending(
                    $user,
                    'zarinpal',
                    $premiumPrice,
                    $authority,
                );

                $zarinpal = new ZarinPalGateway($zarinpalMerchantId, $zarinpalSandbox);
                $callbackUrl = $paymentCallbackBaseUrl . '?gateway=zarinpal';
                $result = $zarinpal->createPayment($premiumPrice, $callbackUrl, 'ارتقا به حساب پریمیوم');

                if (!$result['success']) {
                    safeAnswerCallbackQuery($bot, 'خطا در ایجاد لینک پرداخت!', true);
                    return;
                }

                $realAuthority = $result['authority'] ?? '';
                if (!empty($realAuthority)) {
                    $payment = $paymentService->findByPayload($authority);
                    if ($payment !== null) {
                        $paymentService->setExternalId($payment, $realAuthority);
                    }
                }

                $bot->sendMessage(
                    "برای پرداخت و ارتقا به حساب پریمیوم، روی لینک زیر کلیک کنید:" . PHP_EOL .
                    $result['url'] . PHP_EOL . PHP_EOL .
                    "مبلغ: " . number_format($premiumPrice) . " ریال"
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
        }

        // Actions برای ادمین‌ها
        if (str_starts_with($data, 'action:admin:')) {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند ادمین‌ها را مدیریت کند!', true);
                return;
            }

            $action = str_replace('action:admin:', '', $data);
            
            if ($action === 'list') {
                $admins = $adminService->getAll();
                if (empty($admins)) {
                    $text = "هیچ ادمینی ثبت نشده است.";
                } else {
                    $text = "👥 لیست ادمین‌ها:\n\n";
                    foreach ($admins as $admin) {
                        $role = $admin->isOwner() ? "👑 Owner" : "👤 Admin";
                        $text .= "{$role}: {$admin->getTelegramId()}\n";
                    }
                }
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminsMenu()
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($action === 'add:input') {
                $userStates[$telegramId] = ['action' => 'add_admin', 'step' => 1, 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '➕ افزودن ادمین' . PHP_EOL . PHP_EOL .
                    'لطفاً شناسه تلگرام ادمین جدید را ارسال کنید:' . PHP_EOL .
                    '(مثال: 123456789)',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($action === 'remove:input') {
                $userStates[$telegramId] = ['action' => 'remove_admin', 'step' => 1, 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '🗑️ حذف ادمین' . PHP_EOL . PHP_EOL .
                    'لطفاً شناسه تلگرام ادمینی که می‌خواهید حذف کنید را ارسال کنید:' . PHP_EOL .
                    '(مثال: 123456789)',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
        }

        // Actions برای تبلیغات
        if (str_starts_with($data, 'action:ads:')) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $action = str_replace('action:ads:', '', $data);
            
            // دریافت یا ایجاد admin (اگر owner است)
            $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
            
            if ($admin === null) {
                safeAnswerCallbackQuery($bot, 'خطا در دریافت اطلاعات ادمین!', true);
                return;
            }
            
            if ($action === 'list' || str_starts_with($action, 'list:page:')) {
                $page = 0;
                if (preg_match('/list:page:(\d+)/', $action, $matches)) {
                    $page = (int) $matches[1];
                }
                
                $ads = $adsService->getAllAdsForAdmin($admin->getId());
                if (empty($ads)) {
                    $text = "📢 لیست تبلیغات:\n\nهیچ تبلیغی ثبت نشده است.";
                } else {
                    $currentPage = $page + 1;
                    $text = "📢 لیست تبلیغات (صفحه {$currentPage}):\n\n";
                }
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdsListMenu($ads, $page)
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if (preg_match('/select:(\d+)/', $action, $matches)) {
                $adId = (int) $matches[1];
                $ad = $adsService->findById($adId);
                
                if ($ad === null || $ad->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'تبلیغ یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }
                
                $status = $ad->isEnabled() ? "✅ فعال" : "❌ غیرفعال";
                $text = "📢 اطلاعات تبلیغ #{$adId}\n\n";
                $text .= "وضعیت: {$status}\n";
                $text .= "متن: " . mb_substr($ad->getText(), 0, 200) . (mb_strlen($ad->getText()) > 200 ? "..." : "") . "\n";
                $text .= "فرکانس: هر {$ad->getFrequency()} پیام\n";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdDetailMenu($adId)
                );
                return;
            }

            if (preg_match('/toggle:(\d+)/', $action, $matches)) {
                $adId = (int) $matches[1];
                $ad = $adsService->findById($adId);
                
                if ($ad === null || $ad->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'تبلیغ یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }
                
                $ad->toggle(!$ad->isEnabled());
                $adsService->updateAd($ad);
                
                $status = $ad->isEnabled() ? "فعال" : "غیرفعال";
                safeAnswerCallbackQuery($bot, "تبلیغ {$status} شد ✅");
                
                // به‌روزرسانی منو
                $text = "📢 اطلاعات تبلیغ #{$adId}\n\n";
                $text .= "وضعیت: " . ($ad->isEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                $text .= "متن: " . mb_substr($ad->getText(), 0, 200) . (mb_strlen($ad->getText()) > 200 ? "..." : "") . "\n";
                $text .= "فرکانس: هر {$ad->getFrequency()} پیام\n";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdDetailMenu($adId)
                );
                return;
            }

            if (preg_match('/delete:(\d+)/', $action, $matches)) {
                $adId = (int) $matches[1];
                $ad = $adsService->findById($adId);
                
                if ($ad === null || $ad->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'تبلیغ یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }
                
                $adsService->deleteAd($adId);
                $bot->answerCallbackQuery("تبلیغ #{$adId} حذف شد ✅");
                
                // بازگشت به لیست
                $ads = $adsService->getAllAdsForAdmin($admin->getId());
                if (empty($ads)) {
                    $text = "📢 لیست تبلیغات:\n\nهیچ تبلیغی ثبت نشده است.";
                } else {
                    $text = "📢 لیست تبلیغات:\n\n";
                }
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdsListMenu($ads, 0)
                );
                return;
            }

            if ($action === 'create:input') {
                $userStates[$telegramId] = ['action' => 'create_ad', 'step' => 1, 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '➕ افزودن تبلیغ' . PHP_EOL . PHP_EOL .
                    'لطفاً متن تبلیغ را ارسال کنید:',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
        }

        // Actions برای مدیریت مدیا
        if (str_starts_with($data, 'action:media:')) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            if ($data === 'action:media:list:page:0' || str_starts_with($data, 'action:media:list:page:')) {
                $page = 0;
                if (preg_match('/action:media:list:page:(\d+)/', $data, $matches)) {
                    $page = (int) $matches[1];
                }
                
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($admin === null) {
                    safeAnswerCallbackQuery($bot, 'خطا در دریافت اطلاعات ادمین!', true);
                    return;
                }
                
                $allMedia = $mediaService->listByFolder(1); // در نسخه مینیمال folderId = 1
                
                // فیلتر کردن مدیاهای متعلق به این ادمین
                $userMedia = array_filter($allMedia, fn($m) => $m->getOwnerAdminId() === $admin->getId());
                $userMedia = array_values($userMedia);
                
                if (empty($userMedia)) {
                    $text = "هیچ مدیایی یافت نشد.";
                } else {
                    $currentPage = $page + 1;
                    $text = "📁 لیست مدیاها ({$currentPage}):\n\n";
                }
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildMediaListMenu($userMedia, $page)
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if (preg_match('/action:media:select:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                if ($media === null) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد!', true);
                    return;
                }
                
                $downloads = $analyticsService->getDownloadCount($mediaId);
                $clicks = $analyticsService->getClickCount($mediaId);
                
                $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
                $text .= "نوع: {$media->getType()}\n";
                $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                if ($media->getAutoDeleteSeconds() !== null) {
                    $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
                } else {
                    $text .= "Auto-delete: ❌ غیرفعال\n";
                }
                $text .= "دانلود: {$downloads}\n";
                $text .= "کلیک: {$clicks}\n";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildMediaDetailMenu($mediaId)
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if (preg_match('/action:media:info:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                if ($media === null) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد!', true);
                    return;
                }

                // بررسی دسترسی ادمین
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'شما دسترسی به این مدیا ندارید!', true);
                    return;
                }
                
                // ساخت لینک مدیا
                $code = $linkStore->createForMedia($media);
                
                // دریافت username ربات برای ساخت لینک کامل
                try {
                    $botInfo = $bot->getMe();
                    $botUsername = $botInfo->username ?? '';
                    $deepLink = $botUsername !== '' 
                        ? "https://t.me/{$botUsername}?start={$code}"
                        : "/start {$code}";
                } catch (Throwable) {
                    $deepLink = "/start {$code}";
                }
                
                $downloads = $analyticsService->getDownloadCount($mediaId);
                $clicks = $analyticsService->getClickCount($mediaId);
                
                $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
                $text .= "نوع: {$media->getType()}\n";
                $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                if ($media->getAutoDeleteSeconds() !== null) {
                    $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
                } else {
                    $text .= "Auto-delete: ❌ غیرفعال\n";
                }
                $text .= "دانلود: {$downloads}\n";
                $text .= "کلیک: {$clicks}\n\n";
                $text .= "🔗 <b>لینک مدیا:</b>\n";
                $text .= "<pre>{$code}</pre>\n\n";
                $text .= "🌐 <b>لینک کامل:</b>\n";
                $text .= "<pre>{$deepLink}</pre>";
                
                $keyboard = InlineKeyboardMarkup::make();
                $keyboard->addRow(
                    InlineKeyboardButton::make('◀️ بازگشت', callback_data: "action:media:select:{$mediaId}")
                );
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    $keyboard,
                    'HTML'
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if (preg_match('/action:media:toggle_thumb:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($media === null || $admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }
                
                $media->toggleThumbnail(!$media->isThumbnailEnabled());
                $mediaService->updateMedia($media);
                
                $status = $media->isThumbnailEnabled() ? "فعال" : "غیرفعال";
                $bot->answerCallbackQuery("Thumbnail {$status} شد ✅");
                
                // به‌روزرسانی منو
                $downloads = $analyticsService->getDownloadCount($mediaId);
                $clicks = $analyticsService->getClickCount($mediaId);
                $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
                $text .= "نوع: {$media->getType()}\n";
                $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                if ($media->getAutoDeleteSeconds() !== null) {
                    $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
                } else {
                    $text .= "Auto-delete: ❌ غیرفعال\n";
                }
                $text .= "دانلود: {$downloads}\n";
                $text .= "کلیک: {$clicks}\n";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildMediaDetailMenu($mediaId)
                );
                return;
            }

            if (preg_match('/action:media:toggle_link:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($media === null || $admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }
                
                $media->toggleLinkInCaption(!$media->isLinkInCaptionEnabled());
                $mediaService->updateMedia($media);
                
                $status = $media->isLinkInCaptionEnabled() ? "فعال" : "غیرفعال";
                $bot->answerCallbackQuery("لینک در کپشن {$status} شد ✅");
                
                // به‌روزرسانی منو
                $downloads = $analyticsService->getDownloadCount($mediaId);
                $clicks = $analyticsService->getClickCount($mediaId);
                $text = "📊 اطلاعات مدیا #{$mediaId}\n\n";
                $text .= "نوع: {$media->getType()}\n";
                $text .= "Thumbnail: " . ($media->isThumbnailEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                $text .= "لینک در کپشن: " . ($media->isLinkInCaptionEnabled() ? "✅ فعال" : "❌ غیرفعال") . "\n";
                if ($media->getAutoDeleteSeconds() !== null) {
                    $text .= "Auto-delete: {$media->getAutoDeleteSeconds()} ثانیه\n";
                } else {
                    $text .= "Auto-delete: ❌ غیرفعال\n";
                }
                $text .= "دانلود: {$downloads}\n";
                $text .= "کلیک: {$clicks}\n";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildMediaDetailMenu($mediaId)
                );
                return;
            }

            if (preg_match('/action:media:autodelete:input:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $userStates[$telegramId] = ['action' => 'set_autodelete', 'step' => 1, 'data' => ['media_id' => $mediaId], 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '⏱️ تنظیم Auto-Delete' . PHP_EOL . PHP_EOL .
                    'لطفاً تعداد ثانیه‌ها را ارسال کنید:' . PHP_EOL .
                    '(0 برای غیرفعال کردن)',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            // حذف مدیا - تایید اول
            if (preg_match('/action:media:delete:confirm:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($media === null || $admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }

                $text = "⚠️ هشدار: حذف مدیا" . PHP_EOL . PHP_EOL;
                $text .= "آیا مطمئن هستید که می‌خواهید مدیا #{$mediaId} را حذف کنید؟" . PHP_EOL;
                $text .= "این عمل غیرقابل بازگشت است!";

                $keyboard = InlineKeyboardMarkup::make();
                $keyboard->addRow(
                    InlineKeyboardButton::make('✅ بله، حذف کن', callback_data: "action:media:delete:final:{$mediaId}"),
                    InlineKeyboardButton::make('❌ لغو', callback_data: "action:media:select:{$mediaId}")
                );

                safeEditMessageText($bot, $text, $chatId, $messageId, $keyboard);
                safeAnswerCallbackQuery($bot);
                return;
            }

            // حذف مدیا - تایید نهایی
            if (preg_match('/action:media:delete:final:(\d+)/', $data, $matches)) {
                $mediaId = (int) $matches[1];
                $media = $mediaService->findById($mediaId);
                
                $admin = getOrCreateAdmin($adminService, $telegramId, $ownerTelegramId);
                if ($media === null || $admin === null || $media->getOwnerAdminId() !== $admin->getId()) {
                    safeAnswerCallbackQuery($bot, 'مدیا یافت نشد یا شما دسترسی ندارید!', true);
                    return;
                }

                $mediaService->deleteMedia($mediaId);
                safeAnswerCallbackQuery($bot, "مدیا #{$mediaId} حذف شد ✅");

                // بازگشت به لیست
                $allMedia = $mediaService->listByFolder(1);
                $userMedia = array_filter($allMedia, fn($m) => $m->getOwnerAdminId() === $admin->getId());
                $userMedia = array_values($userMedia);

                if (empty($userMedia)) {
                    $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "هیچ مدیایی یافت نشد. برای افزودن مدیا، یک فایل را ارسال کنید.";
                } else {
                    $text = "📁 مدیریت مدیا" . PHP_EOL . PHP_EOL . "لطفاً مدیای مورد نظر را انتخاب کنید:";
                }

                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildMediaListMenu($userMedia, 0)
                );
                return;
            }
        }

        // Actions برای برودکست
        if (str_starts_with($data, 'action:broadcast:')) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            if ($data === 'action:broadcast:input:all') {
                $userStates[$telegramId] = ['action' => 'broadcast', 'step' => 1, 'data' => ['target' => 'all'], 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '📤 برودکست به همه کاربران' . PHP_EOL . PHP_EOL .
                    'لطفاً شناسه مدیا را ارسال کنید:' . PHP_EOL .
                    '(از منوی مدیریت مدیا می‌توانید شناسه را ببینید)',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($data === 'action:broadcast:input:channel') {
                $userStates[$telegramId] = ['action' => 'broadcast', 'step' => 1, 'data' => ['target' => 'channel'], 'edit_message_id' => $messageId];
                
                safeEditMessageText(
                    $bot,
                    '📤 برودکست به کانال/گروه' . PHP_EOL . PHP_EOL .
                    'لطفاً شناسه مدیا را ارسال کنید:' . PHP_EOL .
                    '(از منوی مدیریت مدیا می‌توانید شناسه را ببینید)',
                    $chatId,
                    $messageId,
                    InlineKeyboardMarkup::make()->addRow(
                        InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                    )
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
        }

        // پاکسازی کلی - تایید اول
        if ($data === 'action:admin:clear_all:confirm:1') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند پاکسازی کلی انجام دهد!', true);
                return;
            }

            $text = "⚠️⚠️ هشدار جدی: پاکسازی کلی" . PHP_EOL . PHP_EOL;
            $text .= "این عمل تمام مدیاها را حذف می‌کند!" . PHP_EOL;
            $text .= "این عمل غیرقابل بازگشت است!" . PHP_EOL . PHP_EOL;
            $text .= "آیا مطمئن هستید؟";

            $keyboard = InlineKeyboardMarkup::make();
            $keyboard->addRow(
                InlineKeyboardButton::make('⚠️ بله، ادامه بده', callback_data: 'action:admin:clear_all:confirm:2'),
                InlineKeyboardButton::make('❌ لغو', callback_data: 'menu:admin')
            );

            safeEditMessageText($bot, $text, $chatId, $messageId, $keyboard);
            safeAnswerCallbackQuery($bot);
            return;
        }

        // پاکسازی کلی - تایید دوم
        if ($data === 'action:admin:clear_all:confirm:2') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند پاکسازی کلی انجام دهد!', true);
                return;
            }

            $text = "⚠️⚠️⚠️ تایید نهایی: پاکسازی کلی" . PHP_EOL . PHP_EOL;
            $text .= "این آخرین فرصت شماست!" . PHP_EOL;
            $text .= "با کلیک روی دکمه زیر، تمام مدیاها حذف خواهند شد!" . PHP_EOL . PHP_EOL;
            $text .= "آیا واقعاً مطمئن هستید؟";

            $keyboard = InlineKeyboardMarkup::make();
            $keyboard->addRow(
                InlineKeyboardButton::make('🗑️ بله، حذف کنم', callback_data: 'action:admin:clear_all:final'),
                InlineKeyboardButton::make('❌ لغو', callback_data: 'menu:admin')
            );

            safeEditMessageText($bot, $text, $chatId, $messageId, $keyboard);
            safeAnswerCallbackQuery($bot);
            return;
        }

        // پاکسازی کلی - اجرای نهایی
        if ($data === 'action:admin:clear_all:final') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند پاکسازی کلی انجام دهد!', true);
                return;
            }

            // دریافت تمام مدیاها و حذف آنها با cascade
            require_once __DIR__ . '/../database-repositories.php';
            $pdo = createDatabaseConnection();
            
            if ($pdo === null) {
                safeAnswerCallbackQuery($bot, 'خطا در اتصال به دیتابیس!', true);
                return;
            }
            
            $allMedia = $mediaService->listByFolder(1);
            $deletedCount = 0;

            foreach ($allMedia as $media) {
                try {
                    // استفاده از cascade delete
                    deleteMediaCascade($pdo, $media->getId());
                    $deletedCount++;
                } catch (Throwable $e) {
                    // در صورت خطا، ادامه می‌دهیم
                    $logService->error('خطا در حذف مدیا در پاکسازی کلی', [
                        'media_id' => $media->getId(),
                    ], $telegramId, null, null, $media->getId(), $e);
                }
            }
            
            // حذف analytics events و logs باقی‌مانده (اگر وجود داشته باشند)
            try {
                $stmt = $pdo->prepare('DELETE FROM analytics_events WHERE media_id IS NOT NULL AND media_id NOT IN (SELECT id FROM media)');
                $stmt->execute();
                
                $stmt = $pdo->prepare('DELETE FROM logs WHERE media_id IS NOT NULL AND media_id NOT IN (SELECT id FROM media)');
                $stmt->execute();
            } catch (Throwable $e) {
                // نادیده می‌گیریم
            }

            $text = "✅ پاکسازی کلی انجام شد!" . PHP_EOL . PHP_EOL;
            $text .= "تعداد مدیاهای حذف شده: {$deletedCount}";

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildAdminMenu(true)
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // Action لغو
        if ($data === 'action:cancel') {
            unset($userStates[$telegramId]);
            
            safeEditMessageText(
                $bot,
                '❌ عملیات لغو شد.',
                $chatId,
                $messageId,
                InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('◀️ بازگشت به منوی اصلی', callback_data: 'menu:main')
                )
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // Actions برای آمار
        if (str_starts_with($data, 'action:stats:')) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $action = str_replace('action:stats:', '', $data);
            
            if ($action === 'general') {
                $totalDownloads = $analyticsService->getDownloadCount();
                $totalClicks = $analyticsService->getClickCount();
                $joinConverted = $analyticsService->getJoinConvertedCount();
                $totalRevenue = $analyticsService->getTotalRevenue();
                $conversionRate = $analyticsService->getConversionRate();
                $totalUsers = $analyticsService->getTotalUniqueUsersCount();

                $text = "📊 آمار کلی ربات" . PHP_EOL . PHP_EOL .
                    "👥 کل کاربران: {$totalUsers}" . PHP_EOL .
                    "📥 کل دانلودها: {$totalDownloads}" . PHP_EOL .
                    "👆 کل کلیک‌ها: {$totalClicks}" . PHP_EOL .
                    "✅ جوین‌های تبدیل شده: {$joinConverted}" . PHP_EOL .
                    "💰 کل پرداخت‌ها: {$totalRevenue}" . PHP_EOL .
                    "📈 نرخ تبدیل کلی: " . number_format($conversionRate, 2) . "%";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminStatsMenu()
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($action === 'users') {
                $text = "👥 آمار کاربران" . PHP_EOL . PHP_EOL;
                $text .= "لطفاً بازه زمانی مورد نظر را انتخاب کنید:";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminUsersStatsMenu()
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if (str_starts_with($action, 'users:')) {
                $period = str_replace('users:', '', $action);
                
                $now = new \DateTimeImmutable('now', new \DateTimeZone('Asia/Tehran'));
                $startDate = null;
                $endDate = $now;
                $periodText = '';
                
                switch ($period) {
                    case 'today':
                        $startDate = $now->setTime(0, 0, 0);
                        $periodText = 'امروز';
                        break;
                    case 'yesterday':
                        $yesterday = $now->modify('-1 day');
                        $startDate = $yesterday->setTime(0, 0, 0);
                        $endDate = $yesterday->setTime(23, 59, 59);
                        $periodText = 'دیروز';
                        break;
                    case 'week':
                        $startDate = $now->modify('monday this week')->setTime(0, 0, 0);
                        $periodText = 'این هفته';
                        break;
                    case 'month':
                        $startDate = $now->modify('first day of this month')->setTime(0, 0, 0);
                        $periodText = 'این ماه';
                        break;
                    case 'year':
                        $startDate = $now->modify('first day of january this year')->setTime(0, 0, 0);
                        $periodText = 'امسال';
                        break;
                    case 'all':
                        $startDate = new \DateTimeImmutable('1970-01-01');
                        $periodText = 'کل';
                        break;
                    default:
                        $startDate = new \DateTimeImmutable('1970-01-01');
                        $periodText = 'کل';
                }
                
                $usersCount = $analyticsService->getUniqueUsersCountInRange($startDate, $endDate);
                $totalUsers = $analyticsService->getTotalUniqueUsersCount();
                
                $text = "👥 آمار کاربران ({$periodText})" . PHP_EOL . PHP_EOL;
                $text .= "👥 تعداد کاربران: {$usersCount}" . PHP_EOL;
                if ($period !== 'all') {
                    $text .= "👥 کل کاربران: {$totalUsers}" . PHP_EOL;
                }
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminUsersStatsMenu()
                );
                safeAnswerCallbackQuery($bot);
                return;
            }

            if ($action === 'revenue') {
                $telegramStars = $analyticsService->getRevenueByGateway('telegram_stars');
                $nextpay = $analyticsService->getRevenueByGateway('nextpay');
                $zarinpal = $analyticsService->getRevenueByGateway('zarinpal');
                $total = $analyticsService->getTotalRevenue();

                $text = "💰 گزارش درآمد" . PHP_EOL . PHP_EOL .
                    "⭐ Telegram Stars: {$telegramStars}" . PHP_EOL .
                    "💳 NextPay: {$nextpay}" . PHP_EOL .
                    "💳 ZarinPal: {$zarinpal}" . PHP_EOL .
                    "📊 کل: {$total}";
                
                safeEditMessageText(
                    $bot,
                    $text,
                    $chatId,
                    $messageId,
                    buildAdminStatsMenu()
                );
                safeAnswerCallbackQuery($bot);
                return;
            }
        }

        // منوی مدیریت لاگ‌ها
        if ($data === 'menu:admin:logs') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $totalLogs = $logService->countLogs();
            $errorLogs = $logService->countLogs('ERROR');
            $warningLogs = $logService->countLogs('WARNING');

            $text = "📝 مدیریت لاگ‌ها" . PHP_EOL . PHP_EOL;
            $text .= "📊 آمار کلی:" . PHP_EOL;
            $text .= "کل لاگ‌ها: {$totalLogs}" . PHP_EOL;
            $text .= "🔴 خطاها: {$errorLogs}" . PHP_EOL;
            $text .= "⚠️ هشدارها: {$warningLogs}";

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildLogsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // نمایش لاگ‌های اخیر
        if ($data === 'action:logs:recent') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $logs = $logService->getLogs(limit: 20);
            
            if (empty($logs)) {
                $text = "📝 لاگ‌های اخیر" . PHP_EOL . PHP_EOL . "هیچ لاگی یافت نشد.";
            } else {
                $text = "📝 لاگ‌های اخیر (20 مورد آخر):" . PHP_EOL . PHP_EOL;
                foreach (array_slice($logs, 0, 10) as $log) {
                    $level = $log['level'] ?? 'UNKNOWN';
                    $message = mb_substr($log['message'] ?? '', 0, 50);
                    $time = date('H:i:s', strtotime($log['created_at'] ?? 'now'));
                    $text .= "[{$time}] [{$level}] {$message}" . PHP_EOL;
                }
                if (count($logs) > 10) {
                    $text .= PHP_EOL . "... و " . (count($logs) - 10) . " لاگ دیگر";
                }
            }

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildLogsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // فیلتر بر اساس سطح
        if (preg_match('/action:logs:level:(DEBUG|INFO|WARNING|ERROR|CRITICAL)/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $level = $matches[1];
            $logs = $logService->getLogs(level: $level, limit: 20);
            $count = $logService->countLogs(level: $level);

            if (empty($logs)) {
                $text = "📝 لاگ‌های سطح {$level}" . PHP_EOL . PHP_EOL . "هیچ لاگی یافت نشد.";
            } else {
                $text = "📝 لاگ‌های سطح {$level} ({$count} مورد):" . PHP_EOL . PHP_EOL;
                foreach (array_slice($logs, 0, 10) as $log) {
                    $message = mb_substr($log['message'] ?? '', 0, 50);
                    $time = date('H:i:s', strtotime($log['created_at'] ?? 'now'));
                    $text .= "[{$time}] {$message}" . PHP_EOL;
                }
                if (count($logs) > 10) {
                    $text .= PHP_EOL . "... و " . (count($logs) - 10) . " لاگ دیگر";
                }
            }

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildLogsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // فیلتر بر اساس بازه زمانی
        if (preg_match('/action:logs:period:(today|yesterday)/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }

            $period = $matches[1];
            $now = new \DateTimeImmutable('now', new \DateTimeZone('Asia/Tehran'));
            
            if ($period === 'today') {
                $startDate = $now->setTime(0, 0, 0);
                $endDate = $now;
                $periodText = 'امروز';
            } else {
                $yesterday = $now->modify('-1 day');
                $startDate = $yesterday->setTime(0, 0, 0);
                $endDate = $yesterday->setTime(23, 59, 59);
                $periodText = 'دیروز';
            }

            $logs = $logService->getLogs(startDate: $startDate, endDate: $endDate, limit: 20);
            $count = $logService->countLogs(startDate: $startDate, endDate: $endDate);

            if (empty($logs)) {
                $text = "📝 لاگ‌های {$periodText}" . PHP_EOL . PHP_EOL . "هیچ لاگی یافت نشد.";
            } else {
                $text = "📝 لاگ‌های {$periodText} ({$count} مورد):" . PHP_EOL . PHP_EOL;
                foreach (array_slice($logs, 0, 10) as $log) {
                    $level = $log['level'] ?? 'UNKNOWN';
                    $message = mb_substr($log['message'] ?? '', 0, 50);
                    $time = date('H:i:s', strtotime($log['created_at'] ?? 'now'));
                    $text .= "[{$time}] [{$level}] {$message}" . PHP_EOL;
                }
                if (count($logs) > 10) {
                    $text .= PHP_EOL . "... و " . (count($logs) - 10) . " لاگ دیگر";
                }
            }

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildLogsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // پاکسازی لاگ‌های قدیمی - تایید
        if ($data === 'action:logs:cleanup:confirm') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند لاگ‌ها را پاک کند!', true);
                return;
            }

            $text = "⚠️ پاکسازی لاگ‌های قدیمی" . PHP_EOL . PHP_EOL;
            $text .= "آیا مطمئن هستید که می‌خواهید لاگ‌های قدیمی‌تر از 30 روز را حذف کنید؟";

            $keyboard = InlineKeyboardMarkup::make();
            $keyboard->addRow(
                InlineKeyboardButton::make('✅ بله، حذف کن', callback_data: 'action:logs:cleanup:final'),
                InlineKeyboardButton::make('❌ لغو', callback_data: 'menu:admin:logs')
            );

            safeEditMessageText($bot, $text, $chatId, $messageId, $keyboard);
            safeAnswerCallbackQuery($bot);
            return;
        }

        // پاکسازی لاگ‌های قدیمی - اجرا
        if ($data === 'action:logs:cleanup:final') {
            if (!$isAdmin || ($ownerTelegramId !== null && $telegramId !== $ownerTelegramId)) {
                safeAnswerCallbackQuery($bot, 'فقط مالک ربات می‌تواند لاگ‌ها را پاک کند!', true);
                return;
            }

            $beforeDate = new \DateTimeImmutable('-30 days', new \DateTimeZone('Asia/Tehran'));
            $deletedCount = $logService->deleteOldLogs($beforeDate);

            $text = "✅ پاکسازی لاگ‌ها انجام شد!" . PHP_EOL . PHP_EOL;
            $text .= "تعداد لاگ‌های حذف شده: {$deletedCount}";

            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildLogsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // مدیریت کانال‌های اجباری
        if ($data === 'menu:admin:forced_join') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            safeEditMessageText(
                $bot,
                '🔗 مدیریت کانال‌های اجباری' . PHP_EOL . PHP_EOL . 'از این بخش می‌توانید کانال‌ها، گروه‌ها یا ربات‌های اجباری را مدیریت کنید.',
                $chatId,
                $messageId,
                buildForcedJoinChannelsMenu()
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // لیست کانال‌های اجباری
        if ($data === 'action:forced_join:list' || preg_match('/^action:forced_join:list:page:(\d+)$/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $page = isset($matches[1]) ? (int) $matches[1] : 0;
            $channels = $forcedJoinChannelService->getAllChannels();
            
            if (empty($channels)) {
                $text = "📋 لیست کانال‌های اجباری:" . PHP_EOL . PHP_EOL . "هیچ کانالی ثبت نشده است.";
            } else {
                $text = "📋 لیست کانال‌های اجباری:" . PHP_EOL . PHP_EOL;
                $start = $page * 5;
                $end = min($start + 5, count($channels));
                $channelsPage = array_slice($channels, $start, 5);
                
                foreach ($channelsPage as $channel) {
                    $status = $channel->isEnabled() ? "✅" : "❌";
                    $text .= "{$status} {$channel->getChannelUsername()} ({$channel->getChannelType()})" . PHP_EOL;
                }
                
                if (count($channels) > 5) {
                    $text .= PHP_EOL . "صفحه " . ($page + 1) . " از " . ceil(count($channels) / 5);
                }
            }
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildForcedJoinChannelsListMenu($channels, $page)
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // افزودن کانال - ورودی
        if ($data === 'action:forced_join:add:input') {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $userStates[$telegramId] = [
                'action' => 'forced_join:add',
                'step' => 1,
            ];
            
            $text = "➕ افزودن کانال اجباری" . PHP_EOL . PHP_EOL;
            $text .= "لطفاً username کانال/گروه را ارسال کنید:" . PHP_EOL;
            $text .= "مثال: @channel_name یا -1001234567890" . PHP_EOL . PHP_EOL;
            $text .= "⚠️ توجه: اگر chat_id ارسال کنید (مثل -1001234567890)، بعد از آن باید invite link کانال را هم ارسال کنید." . PHP_EOL . PHP_EOL;
            $text .= "برای لغو، /cancel را ارسال کنید.";
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                InlineKeyboardMarkup::make()->addRow(
                    InlineKeyboardButton::make('❌ لغو', callback_data: 'action:cancel')
                )
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // انتخاب کانال برای مدیریت
        if (preg_match('/^action:forced_join:select:(\d+)$/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $channelId = (int) $matches[1];
            $channel = $forcedJoinChannelService->findById($channelId);
            
            if ($channel === null) {
                safeAnswerCallbackQuery($bot, 'کانال یافت نشد!', true);
                return;
            }
            
            $status = $channel->isEnabled() ? "✅ فعال" : "❌ غیرفعال";
            $text = "🔗 اطلاعات کانال اجباری" . PHP_EOL . PHP_EOL;
            $text .= "Username: {$channel->getChannelUsername()}" . PHP_EOL;
            $text .= "نوع: {$channel->getChannelType()}" . PHP_EOL;
            $text .= "وضعیت: {$status}";
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildForcedJoinChannelDetailMenu($channelId)
            );
            safeAnswerCallbackQuery($bot);
            return;
        }

        // Toggle کانال
        if (preg_match('/^action:forced_join:toggle:(\d+)$/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $channelId = (int) $matches[1];
            $channel = $forcedJoinChannelService->findById($channelId);
            
            if ($channel === null) {
                safeAnswerCallbackQuery($bot, 'کانال یافت نشد!', true);
                return;
            }
            
            $newStatus = !$channel->isEnabled();
            $forcedJoinChannelService->toggleChannel($channelId, $newStatus);
            
            $status = $newStatus ? "فعال" : "غیرفعال";
            safeAnswerCallbackQuery($bot, "کانال {$status} شد ✅");
            
            // بازگشت به لیست
            $channels = $forcedJoinChannelService->getAllChannels();
            $text = "📋 لیست کانال‌های اجباری:" . PHP_EOL . PHP_EOL;
            if (empty($channels)) {
                $text .= "هیچ کانالی ثبت نشده است.";
            } else {
                foreach (array_slice($channels, 0, 5) as $ch) {
                    $st = $ch->isEnabled() ? "✅" : "❌";
                    $text .= "{$st} {$ch->getChannelUsername()} ({$ch->getChannelType()})" . PHP_EOL;
                }
            }
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildForcedJoinChannelsListMenu($channels, 0)
            );
            return;
        }

        // حذف کانال
        if (preg_match('/^action:forced_join:delete:(\d+)$/', $data, $matches)) {
            if (!$isAdmin) {
                safeAnswerCallbackQuery($bot, 'شما دسترسی ادمین ندارید!', true);
                return;
            }
            
            $channelId = (int) $matches[1];
            $deleted = $forcedJoinChannelService->deleteChannel($channelId);
            
            if (!$deleted) {
                safeAnswerCallbackQuery($bot, 'کانال یافت نشد!', true);
                return;
            }
            
            safeAnswerCallbackQuery($bot, "کانال #{$channelId} حذف شد ✅");
            
            // بازگشت به لیست
            $channels = $forcedJoinChannelService->getAllChannels();
            $text = "📋 لیست کانال‌های اجباری:" . PHP_EOL . PHP_EOL;
            if (empty($channels)) {
                $text .= "هیچ کانالی ثبت نشده است.";
            } else {
                foreach (array_slice($channels, 0, 5) as $ch) {
                    $st = $ch->isEnabled() ? "✅" : "❌";
                    $text .= "{$st} {$ch->getChannelUsername()} ({$ch->getChannelType()})" . PHP_EOL;
                }
            }
            
            safeEditMessageText(
                $bot,
                $text,
                $chatId,
                $messageId,
                buildForcedJoinChannelsListMenu($channels, 0)
            );
            return;
        }

        safeAnswerCallbackQuery($bot);
    });
}
