#!/bin/bash
# -*- coding: utf-8 -*-

# Script to monitor bot and auto-restart if hung
# Usage: ./monitor.sh
# Recommended: Add to crontab: */5 * * * * /path/to/bot/monitor.sh

export LANG=en_US.UTF-8
export LC_ALL=en_US.UTF-8

cd "$(dirname "$0")"

PID_FILE="bot.pid"
FOREVER_PID_FILE="bot.forever.pid"
LOG_FILE="bot.log"
MAX_LOG_AGE=600  # 10 minutes
MAX_CPU_IDLE=300  # 5 minutes with 0% CPU

# بررسی وجود instance های دیگر در حال اجرا
EXISTING_PIDS=$(ps aux | grep -E "[p]hp.*public/index\.php" | awk '{print $2}')

# بررسی اینکه آیا forever script در حال اجراست
FOREVER_RUNNING=false
if [ -f "$FOREVER_PID_FILE" ]; then
    FOREVER_PID=$(cat "$FOREVER_PID_FILE")
    if ps -p "$FOREVER_PID" > /dev/null 2>&1; then
        FOREVER_RUNNING=true
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ℹ️  Forever script is running (PID: $FOREVER_PID)"
        # اگر forever در حال اجراست، monitor نباید start کند
        # فقط وضعیت را چک می‌کنیم
    else
        # Forever PID file وجود دارد اما process نیست
        rm -f "$FOREVER_PID_FILE"
    fi
fi

# اگر forever در حال اجراست، فقط وضعیت را چک می‌کنیم و start نمی‌کنیم
if [ "$FOREVER_RUNNING" = true ]; then
    # بررسی اینکه آیا bot process واقعاً در حال اجراست
    if [ -n "$EXISTING_PIDS" ]; then
        # Bot در حال اجراست
        BOT_PID=$(echo "$EXISTING_PIDS" | head -n 1)
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ✅ Bot is running under forever script (PID: $BOT_PID)"
    else
        # Forever در حال اجراست اما bot process نیست
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Forever script is running but no bot process found"
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ℹ️  Forever script will restart the bot automatically"
    fi
    exit 0
fi

# اگر forever در حال اجرا نیست، منطق عادی monitor را اجرا می‌کنیم
if [ -n "$EXISTING_PIDS" ]; then
    # اگر PID file وجود ندارد اما process در حال اجراست
    if [ ! -f "$PID_FILE" ]; then
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Found running bot processes without PID file: $EXISTING_PIDS"
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Killing orphaned processes..."
        for PID in $EXISTING_PIDS; do
            kill "$PID" 2>/dev/null
        done
        sleep 2
        REMAINING=$(ps aux | grep -E "[p]hp.*public/index\.php" | awk '{print $2}')
        if [ -n "$REMAINING" ]; then
            for PID in $REMAINING; do
                kill -9 "$PID" 2>/dev/null
            done
        fi
        sleep 1
    else
        # اگر PID file وجود دارد، بررسی کنیم که آیا process واقعاً در حال اجراست
        FILE_PID=$(cat "$PID_FILE")
        if ! echo "$EXISTING_PIDS" | grep -q "^$FILE_PID$"; then
            echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  PID file exists but process not found. Cleaning up..."
            rm -f "$PID_FILE"
            # Kill همه process های باقی‌مانده
            for PID in $EXISTING_PIDS; do
                kill "$PID" 2>/dev/null
            done
            sleep 2
            REMAINING=$(ps aux | grep -E "[p]hp.*public/index\.php" | awk '{print $2}')
            if [ -n "$REMAINING" ]; then
                for PID in $REMAINING; do
                    kill -9 "$PID" 2>/dev/null
                done
            fi
            sleep 1
        fi
    fi
fi

# بررسی وجود PID file
if [ ! -f "$PID_FILE" ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Bot is not running (no PID file)"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Attempting to start bot..."
    ./start-background.sh
    exit 0
fi

PID=$(cat "$PID_FILE")

# بررسی وجود پروسه
if ! ps -p "$PID" > /dev/null 2>&1; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Bot process not found (PID: $PID)"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Attempting to restart bot..."
    rm -f "$PID_FILE"
    ./restart.sh
    exit 0
fi

# بررسی zombie process
STATUS=$(ps -p "$PID" -o stat= 2>/dev/null)
if echo "$STATUS" | grep -q "Z"; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Bot process is ZOMBIE (PID: $PID)"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Attempting to restart bot..."
    ./restart.sh
    exit 0
fi

# بررسی stopped process
if echo "$STATUS" | grep -q "T"; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Bot process is STOPPED (PID: $PID)"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Attempting to restart bot..."
    ./restart.sh
    exit 0
fi

# بررسی آخرین فعالیت در لاگ
if [ -f "$LOG_FILE" ]; then
    if command -v stat >/dev/null 2>&1; then
        LOG_MTIME=$(stat -c %Y "$LOG_FILE" 2>/dev/null || stat -f %m "$LOG_FILE" 2>/dev/null)
        CURRENT_TIME=$(date +%s)
        LOG_AGE=$((CURRENT_TIME - LOG_MTIME))
        
        if [ $LOG_AGE -gt $MAX_LOG_AGE ]; then
            echo "[$(date '+%Y-%m-%d %H:%M:%S')] ⚠️  Log file is old ($LOG_AGE seconds, max: $MAX_LOG_AGE)"
            echo "[$(date '+%Y-%m-%d %H:%M:%S')] 🔄 Bot might be hung. Attempting to restart..."
            ./restart.sh
            exit 0
        fi
    fi
fi

# بررسی CPU usage (اگر 0% برای مدت طولانی باشد، ممکن است hang شده باشد)
# این بررسی فقط برای لاگ است، چون CPU ممکن است واقعاً 0% باشد
CPU_USAGE=$(ps -p "$PID" -o %cpu= 2>/dev/null | tr -d ' ')
if [ -n "$CPU_USAGE" ]; then
    CPU_FLOAT=$(echo "$CPU_USAGE" | awk '{print $1}')
    if (( $(echo "$CPU_FLOAT == 0.0" | bc -l 2>/dev/null || echo "0") )); then
        # CPU 0% است - بررسی کنیم که آیا برای مدت طولانی بوده
        # این فقط یک warning است، نه restart
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] ℹ️  Bot CPU usage is 0% (might be idle or hung)"
    fi
fi

# همه چیز خوب است
echo "[$(date '+%Y-%m-%d %H:%M:%S')] ✅ Bot is running normally (PID: $PID)"

