<?php
declare(strict_types=1);

/**
 * Database Migrations
 * Creates all required tables for the uploader bot system
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/logger.php';

/**
 * Run all migrations
 * @return bool
 */
function runMigrations(): bool {
    $db = initDatabase();
    if (!$db) {
        logError("Cannot run migrations: Database connection failed");
        return false;
    }
    
    try {
        $db->beginTransaction();
        
        // Create users table
        createUsersTable($db);
        
        // Create files table
        createFilesTable($db);
        
        // Create admins table
        createAdminsTable($db);
        
        // Create channels table
        createChannelsTable($db);
        
        // Create points table
        createPointsTable($db);
        
        // Create referrals table
        createReferralsTable($db);
        
        // Create settings table
        createSettingsTable($db);
        
        // Create analytics table
        createAnalyticsTable($db);
        
        // Create payments table
        createPaymentsTable($db);
        
        // Create advertisements table
        createAdvertisementsTable($db);
        
        // Create downloads table (for tracking downloads)
        createDownloadsTable($db);
        
        // Create user_states table (for managing user states)
        createUserStatesTable($db);
        
        // Create sent_messages table (for auto-delete)
        createSentMessagesTable($db);
        
        $db->commit();
        logInfo("All migrations completed successfully");
        return true;
    } catch (PDOException $e) {
        $db->rollBack();
        logError("Migration failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Create users table
 */
function createUsersTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS users (
        id BIGINT UNSIGNED PRIMARY KEY,
        username VARCHAR(255) NULL,
        first_name VARCHAR(255) NOT NULL,
        last_name VARCHAR(255) NULL,
        is_blocked TINYINT(1) DEFAULT 0,
        referral_code VARCHAR(50) UNIQUE NULL,
        referred_by BIGINT UNSIGNED NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_username (username),
        INDEX idx_referral_code (referral_code),
        INDEX idx_referred_by (referred_by)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create files table
 */
function createFilesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS files (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        file_id VARCHAR(255) NOT NULL,
        file_code VARCHAR(50) UNIQUE NOT NULL,
        file_type ENUM('photo', 'video', 'document', 'audio', 'voice', 'video_note', 'sticker', 'animation') NOT NULL,
        file_size BIGINT UNSIGNED NULL,
        caption TEXT NULL,
        thumbnail_file_id VARCHAR(255) NULL,
        has_thumbnail TINYINT(1) DEFAULT 1,
        has_link TINYINT(1) DEFAULT 1,
        uploaded_by BIGINT UNSIGNED NOT NULL,
        forced_join_channels JSON NULL,
        auto_delete_at TIMESTAMP NULL,
        is_active TINYINT(1) DEFAULT 1,
        download_count INT UNSIGNED DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_file_id (file_id),
        INDEX idx_file_code (file_code),
        INDEX idx_uploaded_by (uploaded_by),
        INDEX idx_auto_delete_at (auto_delete_at),
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Add file_code column if it doesn't exist (for existing databases)
    try {
        $db->exec("ALTER TABLE files ADD COLUMN file_code VARCHAR(50) UNIQUE NULL AFTER file_id");
        $db->exec("ALTER TABLE files ADD INDEX idx_file_code (file_code)");
        
        // Generate file_code for existing files
        $existing_files = $db->query("SELECT id FROM files WHERE file_code IS NULL")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($existing_files as $file) {
            require_once __DIR__ . '/validators.php';
            $file_code = generateFileCode();
            $db->prepare("UPDATE files SET file_code = ? WHERE id = ?")->execute([$file_code, $file['id']]);
        }
        
        // Make file_code NOT NULL after populating
        $db->exec("ALTER TABLE files MODIFY COLUMN file_code VARCHAR(50) NOT NULL");
    } catch (PDOException $e) {
        // Column might already exist, ignore error
    }
}

/**
 * Create admins table
 */
function createAdminsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS admins (
        id BIGINT UNSIGNED PRIMARY KEY,
        added_by BIGINT UNSIGNED NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_added_by (added_by)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create channels table
 */
function createChannelsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS channels (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        channel_id BIGINT NOT NULL,
        channel_username VARCHAR(255) NULL,
        channel_title VARCHAR(255) NOT NULL,
        channel_link VARCHAR(500) NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_channel_id (channel_id),
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create points table
 */
function createPointsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS points (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        points INT NOT NULL DEFAULT 0,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY unique_user (user_id),
        INDEX idx_user_id (user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create referrals table
 */
function createReferralsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS referrals (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        referrer_id BIGINT UNSIGNED NOT NULL,
        referred_id BIGINT UNSIGNED NOT NULL,
        points_given_referrer INT DEFAULT 0,
        points_given_referred INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_referrer_id (referrer_id),
        INDEX idx_referred_id (referred_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create settings table
 */
function createSettingsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS settings (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_setting_key (setting_key)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Insert default settings
    $defaultSettings = [
        ['is_paid_mode', '0'],
        ['forced_join_text', 'لطفاً ابتدا در کانال‌های زیر عضو شوید:'],
        ['mandatory_seen_enabled', '0'],
        ['stars_to_points_rate', '1'], // 1 star = 1 point
        ['referral_points_referrer', '5'],
        ['referral_points_referred', '3'],
    ];
    
    $stmt = $db->prepare("INSERT IGNORE INTO settings (setting_key, setting_value) VALUES (?, ?)");
    foreach ($defaultSettings as $setting) {
        $stmt->execute($setting);
    }
}

/**
 * Create analytics table
 */
function createAnalyticsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS analytics (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        event_type VARCHAR(50) NOT NULL,
        user_id BIGINT UNSIGNED NULL,
        file_id INT UNSIGNED NULL,
        metadata JSON NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_event_type (event_type),
        INDEX idx_user_id (user_id),
        INDEX idx_file_id (file_id),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create payments table
 */
function createPaymentsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS payments (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        stars_amount INT UNSIGNED NOT NULL,
        points_amount INT UNSIGNED NOT NULL,
        status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
        telegram_payment_id VARCHAR(255) NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_status (status),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create advertisements table
 */
function createAdvertisementsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS advertisements (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        media_type VARCHAR(50) NULL,
        media_file_id VARCHAR(255) NULL,
        is_active TINYINT(1) DEFAULT 1,
        show_count INT UNSIGNED DEFAULT 0,
        click_count INT UNSIGNED DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create downloads table
 */
function createDownloadsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS downloads (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        file_id INT UNSIGNED NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_file_id (file_id),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create user_states table
 */
function createUserStatesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS user_states (
        user_id BIGINT UNSIGNED PRIMARY KEY,
        state VARCHAR(100) NULL,
        state_data JSON NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create sent_messages table (for auto-delete)
 */
function createSentMessagesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS sent_messages (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        chat_id BIGINT NOT NULL,
        message_id INT NOT NULL,
        file_id INT UNSIGNED NULL,
        delete_at TIMESTAMP NOT NULL,
        is_deleted TINYINT(1) DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_chat_id (chat_id),
        INDEX idx_delete_at (delete_at),
        INDEX idx_is_deleted (is_deleted)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

