<?php
declare(strict_types=1);

/**
 * User Management Handler
 * Handles user management operations
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';
require_once __DIR__ . '/point-handler.php';
require_once __DIR__ . '/referral-handler.php';

/**
 * Search users
 * @param string $query Search query (username, name, or user ID)
 * @return array|false
 */
function searchUsers(string $query): array|false {
    $query = trim($query);
    
    // If query is numeric, search by ID
    if (is_numeric($query)) {
        $user_id = validateUserId($query);
        if ($user_id === false) {
            return false;
        }
        
        $user = dbQueryOne(
            "SELECT * FROM users WHERE id = ?",
            [$user_id]
        );
        
        return $user !== false ? [$user] : false;
    }
    
    // Search by username or name
    $search_term = "%{$query}%";
    
    $users = dbQuery(
        "SELECT * FROM users 
         WHERE username LIKE ? OR first_name LIKE ? OR last_name LIKE ?
         LIMIT 20",
        [$search_term, $search_term, $search_term]
    );
    
    return $users;
}

/**
 * Get user information
 * @param int $user_id
 * @return array|false
 */
function getUserInformation(int $user_id): array|false {
    $user = dbQueryOne(
        "SELECT * FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    // Get points
    $points = getUserPoints($user_id);
    
    // Get download count
    $downloads = dbQueryOne(
        "SELECT COUNT(*) as count FROM downloads WHERE user_id = ?",
        [$user_id]
    );
    $download_count = (int)($downloads['count'] ?? 0);
    
    // Get referral stats
    $referral_stats = getReferralStats($user_id);
    
    return [
        'user' => $user,
        'points' => $points,
        'download_count' => $download_count,
        'referral_stats' => $referral_stats
    ];
}

/**
 * Display user information
 * @param int $chat_id
 * @param int $user_id
 * @return void
 */
function displayUserInfo(int $chat_id, int $user_id): void {
    $info = getUserInformation($user_id);
    
    if ($info === false) {
        sendMessage($chat_id, "❌ کاربر یافت نشد.");
        return;
    }
    
    $user = $info['user'];
    $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
    $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
    
    $message = "👤 اطلاعات کاربر:\n\n";
    $message .= "👤 نام: {$name}\n";
    $message .= "🆔 شناسه: {$user['id']}\n";
    $message .= "📱 یوزرنیم: {$username}\n";
    $message .= "💎 امتیاز: {$info['points']}\n";
    $message .= "📥 دانلودها: {$info['download_count']}\n";
    $message .= "📅 تاریخ عضویت: {$user['created_at']}\n";
    $message .= "🚫 وضعیت: " . ($user['is_blocked'] ? 'بلاک شده' : 'عادی') . "\n";
    
    if ($info['referral_stats']['total_referrals'] > 0) {
        $message .= "\n📊 آمار معرف:\n";
        $message .= "   👥 معرف‌ها: {$info['referral_stats']['total_referrals']}\n";
        $message .= "   💎 امتیاز کسب شده: {$info['referral_stats']['total_points_earned']}\n";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Handle search user command
 * @param array $message
 * @return void
 */
function handleSearchUser(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کاربر جستجو کنند.");
        return;
    }
    
    // Extract search query
    $text = $message['text'] ?? '';
    $query = preg_replace('/^\/\w+\s*/', '', $text);
    $query = trim($query);
    
    if (empty($query)) {
        sendMessage($chat_id, "❌ لطفاً یوزرنیم، نام یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    $users = searchUsers($query);
    
    if ($users === false || empty($users)) {
        sendMessage($chat_id, "❌ کاربری یافت نشد.");
        return;
    }
    
    if (count($users) === 1) {
        // Show detailed info for single result
        displayUserInfo($chat_id, (int)$users[0]['id']);
    } else {
        // Show list of results
        $message = "🔍 نتایج جستجو:\n\n";
        
        foreach ($users as $index => $user) {
            $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
            $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
            
            $message .= ($index + 1) . ". {$name}\n";
            $message .= "   🆔 {$user['id']}\n";
            $message .= "   📱 {$username}\n\n";
        }
        
        sendMessage($chat_id, $message);
    }
}

