<?php
declare(strict_types=1);

/**
 * File Upload Handler
 * Handles file uploads (single and album) by admins
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Handle single file upload
 * @param array $message Telegram message object
 * @return void
 */
function handleFileUpload(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $user_data = $message['from'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند فایل آپلود کنند.");
        return;
    }
    
    // Ensure user exists
    ensureUserExists($user_id, $user_data);
    
    // Extract file information
    $file_info = extractFileInfo($message);
    
    if ($file_info === false) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات فایل.");
        return;
    }
    
    // Save file to database
    $file_id_db = saveFileToDatabase($file_info, $user_id);
    
    if ($file_id_db === false) {
        sendMessage($chat_id, "❌ خطا در ذخیره فایل در دیتابیس.");
        return;
    }
    
    // Log analytics
    logAnalytics('file_uploaded', $user_id, $file_id_db, [
        'file_type' => $file_info['file_type'],
        'file_size' => $file_info['file_size'] ?? 0
    ]);
    
    // Get file code for download link
    $file = getFileById($file_id_db);
    $file_code = $file['file_code'] ?? '';
    
    // Get bot username
    $bot_username = BOT_USERNAME ?? '';
    if (empty($bot_username)) {
        // Try to get from bot info
        $bot_info = bot('getMe', []);
        if ($bot_info !== false && isset($bot_info['ok']) && $bot_info['ok']) {
            $bot_username = $bot_info['result']['username'] ?? '';
        }
    }
    
    // Create download link
    $download_link = '';
    if (!empty($bot_username) && !empty($file_code)) {
        $download_link = "https://t.me/{$bot_username}?start={$file_code}";
    }
    
    $message = "✅ فایل با موفقیت آپلود شد!\n";
    $message .= "📁 شناسه فایل: {$file_id_db}\n";
    if (!empty($download_link)) {
        $message .= "\n🔗 لینک دانلود:\n{$download_link}";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Handle album upload
 * @param array $message Telegram message object
 * @return void
 */
function handleAlbumUpload(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $user_data = $message['from'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند فایل آپلود کنند.");
        return;
    }
    
    // Ensure user exists
    ensureUserExists($user_id, $user_data);
    
    // Extract album files
    $media_group_id = $message['media_group_id'] ?? null;
    
    if (!$media_group_id) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات آلبوم.");
        return;
    }
    
    // Note: Telegram sends album files in separate updates
    // We need to handle this by storing temporary data
    // For now, we'll handle each file individually
    $file_info = extractFileInfo($message);
    
    if ($file_info === false) {
        return;
    }
    
    // Save file to database with media_group_id
    $file_id_db = saveFileToDatabase($file_info, $user_id, $media_group_id);
    
    if ($file_id_db === false) {
        return;
    }
    
    // Log analytics
    logAnalytics('file_uploaded', $user_id, $file_id_db, [
        'file_type' => $file_info['file_type'],
        'media_group_id' => $media_group_id,
        'is_album' => true
    ]);
}

/**
 * Extract file information from Telegram message
 * @param array $message
 * @return array|false
 */
function extractFileInfo(array $message): array|false {
    $file_id = null;
    $file_type = null;
    $file_size = null;
    $caption = $message['caption'] ?? null;
    $thumbnail_file_id = null;
    
    // Check for photo
    if (isset($message['photo']) && is_array($message['photo'])) {
        // Get largest photo
        $photos = $message['photo'];
        $largest_photo = end($photos);
        $file_id = $largest_photo['file_id'] ?? null;
        $file_type = 'photo';
        $file_size = $largest_photo['file_size'] ?? null;
    }
    // Check for video
    elseif (isset($message['video'])) {
        $file_id = $message['video']['file_id'] ?? null;
        $file_type = 'video';
        $file_size = $message['video']['file_size'] ?? null;
        $thumbnail_file_id = $message['video']['thumb']['file_id'] ?? null;
    }
    // Check for document
    elseif (isset($message['document'])) {
        $file_id = $message['document']['file_id'] ?? null;
        $file_type = 'document';
        $file_size = $message['document']['file_size'] ?? null;
        $thumbnail_file_id = $message['document']['thumb']['file_id'] ?? null;
    }
    // Check for audio
    elseif (isset($message['audio'])) {
        $file_id = $message['audio']['file_id'] ?? null;
        $file_type = 'audio';
        $file_size = $message['audio']['file_size'] ?? null;
    }
    // Check for voice
    elseif (isset($message['voice'])) {
        $file_id = $message['voice']['file_id'] ?? null;
        $file_type = 'voice';
        $file_size = $message['voice']['file_size'] ?? null;
    }
    // Check for video_note
    elseif (isset($message['video_note'])) {
        $file_id = $message['video_note']['file_id'] ?? null;
        $file_type = 'video_note';
        $file_size = $message['video_note']['file_size'] ?? null;
    }
    // Check for sticker
    elseif (isset($message['sticker'])) {
        $file_id = $message['sticker']['file_id'] ?? null;
        $file_type = 'sticker';
        $file_size = $message['sticker']['file_size'] ?? null;
    }
    // Check for animation
    elseif (isset($message['animation'])) {
        $file_id = $message['animation']['file_id'] ?? null;
        $file_type = 'animation';
        $file_size = $message['animation']['file_size'] ?? null;
        $thumbnail_file_id = $message['animation']['thumb']['file_id'] ?? null;
    }
    
    if ($file_id === null || $file_type === null) {
        return false;
    }
    
    return [
        'file_id' => $file_id,
        'file_type' => $file_type,
        'file_size' => $file_size,
        'caption' => $caption,
        'thumbnail_file_id' => $thumbnail_file_id
    ];
}

/**
 * Save file to database
 * @param array $file_info
 * @param int $uploaded_by
 * @param string|null $media_group_id
 * @return int|false File ID in database
 */
function saveFileToDatabase(array $file_info, int $uploaded_by, ?string $media_group_id = null): int|false {
    $file_id = $file_info['file_id'];
    $file_type = $file_info['file_type'];
    $file_size = $file_info['file_size'];
    $caption = $file_info['caption'];
    $thumbnail_file_id = $file_info['thumbnail_file_id'];
    
    // Validate file type
    if (!validateFileType($file_type)) {
        return false;
    }
    
    // Generate unique file code
    $file_code = generateFileCode();
    
    $result = dbExecute(
        "INSERT INTO files (file_id, file_code, file_type, file_size, caption, thumbnail_file_id, uploaded_by) 
         VALUES (?, ?, ?, ?, ?, ?, ?)",
        [$file_id, $file_code, $file_type, $file_size, $caption, $thumbnail_file_id, $uploaded_by]
    );
    
    if ($result === false) {
        return false;
    }
    
    $db_file_id = dbLastInsertId();
    return $db_file_id ? (int)$db_file_id : false;
}

/**
 * Get file by database ID
 * @param int $file_id
 * @return array|false
 */
function getFileById(int $file_id): array|false {
    return dbQueryOne(
        "SELECT * FROM files WHERE id = ? AND is_active = 1",
        [$file_id]
    );
}

/**
 * Get file by file code
 * @param string $file_code
 * @return array|false
 */
function getFileByCode(string $file_code): array|false {
    return dbQueryOne(
        "SELECT * FROM files WHERE file_code = ? AND is_active = 1",
        [$file_code]
    );
}

/**
 * Get all files
 * @param int $limit
 * @param int $offset
 * @return array|false
 */
function getAllFiles(int $limit = 50, int $offset = 0): array|false {
    return dbQuery(
        "SELECT * FROM files WHERE is_active = 1 ORDER BY created_at DESC LIMIT ? OFFSET ?",
        [$limit, $offset]
    );
}

/**
 * Delete file (soft delete)
 * @param int $file_id
 * @return bool
 */
function deleteFile(int $file_id): bool {
    $result = dbExecute(
        "UPDATE files SET is_active = 0 WHERE id = ?",
        [$file_id]
    );
    
    return $result !== false;
}

/**
 * Log analytics event
 * @param string $event_type
 * @param int|null $user_id
 * @param int|null $file_id
 * @param array|null $metadata
 * @return void
 */
function logAnalytics(string $event_type, ?int $user_id = null, ?int $file_id = null, ?array $metadata = null): void {
    $metadata_json = $metadata ? json_encode($metadata, JSON_UNESCAPED_UNICODE) : null;
    
    dbExecute(
        "INSERT INTO analytics (event_type, user_id, file_id, metadata) VALUES (?, ?, ?, ?)",
        [$event_type, $user_id, $file_id, $metadata_json]
    );
}

