<?php
declare(strict_types=1);

/**
 * Settings Handler
 * Handles bot settings and configuration
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/logger.php';
require_once __DIR__ . '/referral-handler.php';

/**
 * Get setting value
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function getSettingValue(string $key, mixed $default = null): mixed {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if ($setting === false) {
        return $default;
    }
    
    $value = $setting['setting_value'];
    
    // Convert to boolean if needed
    if ($value === '0' || $value === '1') {
        return (bool)(int)$value;
    }
    
    // Convert to int if numeric
    if (is_numeric($value)) {
        return (int)$value;
    }
    
    return $value ?: $default;
}

/**
 * Set setting value
 * @param string $key
 * @param mixed $value
 * @return bool
 */
function setSettingValue(string $key, mixed $value): bool {
    // Convert boolean to string
    if (is_bool($value)) {
        $value = $value ? '1' : '0';
    } else {
        $value = (string)$value;
    }
    
    $result = dbExecute(
        "UPDATE settings SET setting_value = ? WHERE setting_key = ?",
        [$value, $key]
    );
    
    if ($result === false) {
        // Insert if doesn't exist
        return dbExecute(
            "INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)",
            [$key, $value]
        ) !== false;
    }
    
    return true;
}

/**
 * Check if bot is in paid mode
 * @return bool
 */
function isPaidMode(): bool {
    return (bool)getSettingValue('is_paid_mode', false);
}

/**
 * Toggle paid mode
 * @param bool $enabled
 * @return bool
 */
function togglePaidMode(bool $enabled): bool {
    return setSettingValue('is_paid_mode', $enabled);
}

/**
 * Check if mandatory seen is enabled
 * @return bool
 */
function isMandatorySeenEnabled(): bool {
    return (bool)getSettingValue('mandatory_seen_enabled', false);
}

/**
 * Toggle mandatory seen
 * @param bool $enabled
 * @return bool
 */
function toggleMandatorySeen(bool $enabled): bool {
    return setSettingValue('mandatory_seen_enabled', $enabled);
}

/**
 * Send message with mandatory seen
 * @param int $chat_id
 * @param string $text
 * @param array|null $keyboard
 * @return array|false
 */
function sendMessageWithMandatorySeen(int $chat_id, string $text, ?array $keyboard = null): array|false {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'message_effect_id' => '👀' // Mandatory seen effect
    ];
    
    if ($keyboard) {
        $data['reply_markup'] = json_encode($keyboard, JSON_UNESCAPED_UNICODE);
    }
    
    return bot('sendMessage', $data);
}

/**
 * Display settings
 * @param int $chat_id
 * @return void
 */
function displaySettings(int $chat_id): void {
    $paid_mode = isPaidMode();
    $mandatory_seen = isMandatorySeenEnabled();
    $forced_join_text = getForcedJoinText();
    $stars_rate = getSettingValue('stars_to_points_rate', 1);
    $referral_points_referrer = getSettingValue('referral_points_referrer', 5);
    $referral_points_referred = getSettingValue('referral_points_referred', 3);
    
    $message = "⚙️ تنظیمات ربات:\n\n";
    $message .= "💰 حالت پولی: " . ($paid_mode ? '✅ فعال' : '❌ غیرفعال') . "\n";
    $message .= "👁 Mandatory Seen: " . ($mandatory_seen ? '✅ فعال' : '❌ غیرفعال') . "\n";
    $message .= "📢 متن Forced Join: {$forced_join_text}\n";
    $message .= "⭐ نرخ Stars به امتیاز: {$stars_rate}\n";
    $message .= "🎁 امتیاز معرف (معرف): {$referral_points_referrer}\n";
    $message .= "🎁 امتیاز معرف (معرف‌شده): {$referral_points_referred}\n";
    
    sendMessage($chat_id, $message);
}

/**
 * Handle update forced join text
 * @param array $message
 * @return void
 */
function handleUpdateForcedJoinText(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند تنظیمات را تغییر دهند.");
        return;
    }
    
    // Extract new text
    $text = $message['text'] ?? '';
    $new_text = preg_replace('/^\/\w+\s*/', '', $text);
    $new_text = trim($new_text);
    
    if (empty($new_text)) {
        sendMessage($chat_id, "❌ لطفاً متن جدید را ارسال کنید.");
        return;
    }
    
    if (updateForcedJoinText($new_text)) {
        sendMessage($chat_id, "✅ متن Forced Join با موفقیت به‌روزرسانی شد!");
    } else {
        sendMessage($chat_id, "❌ خطا در به‌روزرسانی متن.");
    }
}

