<?php
declare(strict_types=1);

/**
 * Referral System Handler
 * Handles referral system for earning points
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';
require_once __DIR__ . '/point-handler.php';

/**
 * Get or create referral code for user
 * @param int $user_id
 * @return string|false
 */
function getUserReferralCode(int $user_id): string|false {
    $user = dbQueryOne(
        "SELECT referral_code FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    // If user already has referral code, return it
    if (!empty($user['referral_code'])) {
        return $user['referral_code'];
    }
    
    // Generate new referral code
    $code = generateReferralCode($user_id);
    
    // Save to database
    $result = dbExecute(
        "UPDATE users SET referral_code = ? WHERE id = ?",
        [$code, $user_id]
    );
    
    if ($result === false) {
        return false;
    }
    
    return $code;
}

/**
 * Handle referral link usage
 * @param int $user_id New user ID
 * @param string $referral_code Referral code
 * @return array ['success' => bool, 'referrer_id' => int|null, 'points_given' => array]
 */
function handleReferralUsage(int $user_id, string $referral_code): array {
    // Find referrer
    $referrer = dbQueryOne(
        "SELECT id FROM users WHERE referral_code = ?",
        [$referral_code]
    );
    
    if ($referrer === false) {
        return ['success' => false, 'referrer_id' => null, 'points_given' => []];
    }
    
    $referrer_id = (int)$referrer['id'];
    
    // Check if user was already referred
    $existing = dbQueryOne(
        "SELECT id FROM referrals WHERE referred_id = ?",
        [$user_id]
    );
    
    if ($existing !== false) {
        return ['success' => false, 'referrer_id' => $referrer_id, 'points_given' => []];
    }
    
    // Get referral points from settings
    $points_referrer = getSetting('referral_points_referrer', 5);
    $points_referred = getSetting('referral_points_referred', 3);
    
    // Give points to referrer
    addUserPoints($referrer_id, $points_referrer);
    
    // Give points to referred user
    addUserPoints($user_id, $points_referred);
    
    // Record referral
    dbExecute(
        "INSERT INTO referrals (referrer_id, referred_id, points_given_referrer, points_given_referred) 
         VALUES (?, ?, ?, ?)",
        [$referrer_id, $user_id, $points_referrer, $points_referred]
    );
    
    // Update user's referred_by
    dbExecute(
        "UPDATE users SET referred_by = ? WHERE id = ?",
        [$referrer_id, $user_id]
    );
    
    // Log analytics
    logAnalytics('referral_used', $user_id, null, [
        'referrer_id' => $referrer_id,
        'points_referrer' => $points_referrer,
        'points_referred' => $points_referred
    ]);
    
    return [
        'success' => true,
        'referrer_id' => $referrer_id,
        'points_given' => [
            'referrer' => $points_referrer,
            'referred' => $points_referred
        ]
    ];
}

/**
 * Get referral statistics for user
 * @param int $user_id
 * @return array
 */
function getReferralStats(int $user_id): array {
    $referrals = dbQuery(
        "SELECT COUNT(*) as total_referrals, 
                SUM(points_given_referrer) as total_points_earned 
         FROM referrals WHERE referrer_id = ?",
        [$user_id]
    );
    
    $stats = [
        'total_referrals' => 0,
        'total_points_earned' => 0
    ];
    
    if ($referrals !== false && !empty($referrals)) {
        $stats['total_referrals'] = (int)$referrals[0]['total_referrals'];
        $stats['total_points_earned'] = (int)($referrals[0]['total_points_earned'] ?? 0);
    }
    
    return $stats;
}

/**
 * Display referral link and stats
 * @param int $chat_id
 * @param int $user_id
 * @return void
 */
function displayReferralInfo(int $chat_id, int $user_id): void {
    $code = getUserReferralCode($user_id);
    
    if ($code === false) {
        sendMessage($chat_id, "❌ خطا در ایجاد کد معرف.");
        return;
    }
    
    $bot_username = BOT_USERNAME ?? 'your_bot';
    $referral_link = "https://t.me/{$bot_username}?start={$code}";
    
    $stats = getReferralStats($user_id);
    
    $message = "🔗 لینک معرف شما:\n";
    $message .= "{$referral_link}\n\n";
    $message .= "📊 آمار معرف:\n";
    $message .= "👥 تعداد معرف‌ها: {$stats['total_referrals']}\n";
    $message .= "💎 امتیاز کسب شده: {$stats['total_points_earned']}\n\n";
    $message .= "💡 با هر معرف، شما و معرف‌شده امتیاز دریافت می‌کنید!";
    
    sendMessage($chat_id, $message);
}

/**
 * Get setting value
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function getSetting(string $key, mixed $default = null): mixed {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if ($setting === false) {
        return $default;
    }
    
    $value = $setting['setting_value'];
    
    // Try to convert to int if numeric
    if (is_numeric($value)) {
        return (int)$value;
    }
    
    return $value ?: $default;
}

