<?php
declare(strict_types=1);

/**
 * Point System Handler
 * Handles point management (1 point = 1 download)
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Get user points
 * @param int $user_id
 * @return int
 */
function getUserPoints(int $user_id): int {
    $result = dbQueryOne(
        "SELECT points FROM points WHERE user_id = ?",
        [$user_id]
    );
    
    if ($result === false) {
        // Initialize points for user
        dbExecute(
            "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
            [$user_id]
        );
        return 0;
    }
    
    return (int)$result['points'];
}

/**
 * Add points to user
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function addUserPoints(int $user_id, int $points): bool {
    if ($points <= 0) {
        return false;
    }
    
    // Ensure user has points record
    dbExecute(
        "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
        [$user_id]
    );
    
    $result = dbExecute(
        "UPDATE points SET points = points + ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

/**
 * Deduct points from user
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function deductUserPoints(int $user_id, int $points): bool {
    if ($points <= 0) {
        return false;
    }
    
    $current_points = getUserPoints($user_id);
    
    if ($current_points < $points) {
        return false; // Not enough points
    }
    
    $result = dbExecute(
        "UPDATE points SET points = points - ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

/**
 * Check if user has enough points
 * @param int $user_id
 * @param int $required_points
 * @return bool
 */
function hasEnoughPoints(int $user_id, int $required_points): bool {
    $current_points = getUserPoints($user_id);
    return $current_points >= $required_points;
}

/**
 * Handle download with point deduction
 * @param int $user_id
 * @param int $file_id
 * @param int $chat_id
 * @return array ['success' => bool, 'message' => string, 'sent_message_id' => int|null]
 */
function handleFileDownload(int $user_id, int $file_id, int $chat_id): array {
    require_once __DIR__ . '/auto-delete-handler.php';
    
    // Check if file exists
    $file = getFileById($file_id);
    
    if ($file === false) {
        return ['success' => false, 'message' => '❌ فایل یافت نشد.', 'sent_message_id' => null];
    }
    
    // Check if user has enough points
    if (!hasEnoughPoints($user_id, 1)) {
        return [
            'success' => false,
            'message' => "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.",
            'sent_message_id' => null
        ];
    }
    
    // Deduct 1 point
    if (!deductUserPoints($user_id, 1)) {
        return ['success' => false, 'message' => '❌ خطا در کسر امتیاز.', 'sent_message_id' => null];
    }
    
    // Send file to user based on file type
    $sent_message_id = null;
    $result = false;
    
    $file_id_telegram = $file['file_id'];
    $caption = $file['caption'] ?? null;
    
    switch ($file['file_type']) {
        case 'photo':
            $result = bot('sendPhoto', [
                'chat_id' => $chat_id,
                'photo' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
            
        case 'video':
            $result = bot('sendVideo', [
                'chat_id' => $chat_id,
                'video' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
            
        case 'document':
            $result = bot('sendDocument', [
                'chat_id' => $chat_id,
                'document' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
            
        case 'audio':
            $result = bot('sendAudio', [
                'chat_id' => $chat_id,
                'audio' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
            
        case 'voice':
            $result = bot('sendVoice', [
                'chat_id' => $chat_id,
                'voice' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
            
        case 'video_note':
            $result = bot('sendVideoNote', [
                'chat_id' => $chat_id,
                'video_note' => $file_id_telegram
            ]);
            break;
            
        case 'sticker':
            $result = bot('sendSticker', [
                'chat_id' => $chat_id,
                'sticker' => $file_id_telegram
            ]);
            break;
            
        case 'animation':
            $result = bot('sendAnimation', [
                'chat_id' => $chat_id,
                'animation' => $file_id_telegram,
                'caption' => $caption
            ]);
            break;
    }
    
    if ($result !== false && isset($result['ok']) && $result['ok']) {
        $sent_message_id = (int)$result['result']['message_id'];
        
        // Check if auto-delete is set for this file
        $auto_delete_timer = getFileAutoDeleteTimer($file_id);
        if ($auto_delete_timer !== false && $auto_delete_timer > 0) {
            // IMPORTANT: Only delete the message from user's chat, NOT from database!
            // The file remains in database and can be downloaded again by other users.
            // Delete message after timer expires (in background)
            // Use ignore_user_abort and set_time_limit to continue execution after webhook response
            ignore_user_abort(true);
            set_time_limit(0); // No time limit
            
            // Start background deletion process
            if (function_exists('fastcgi_finish_request')) {
                fastcgi_finish_request(); // Send response to client immediately
            }
            
            // Wait for timer (in background)
            sleep($auto_delete_timer);
            
            // Delete ONLY the message from user's chat (NOT from database)
            bot('deleteMessage', [
                'chat_id' => $chat_id,
                'message_id' => $sent_message_id
            ]);
            
            logInfo("Auto-deleted message: {$sent_message_id} from chat {$chat_id} after {$auto_delete_timer} seconds (file remains in database)");
        }
    }
    
    // Record download
    dbExecute(
        "INSERT INTO downloads (user_id, file_id) VALUES (?, ?)",
        [$user_id, $file_id]
    );
    
    // Update file download count
    dbExecute(
        "UPDATE files SET download_count = download_count + 1 WHERE id = ?",
        [$file_id]
    );
    
    // Log analytics
    logAnalytics('file_downloaded', $user_id, $file_id, [
        'file_type' => $file['file_type']
    ]);
    
    return [
        'success' => true,
        'message' => "✅ فایل با موفقیت دانلود شد!\n💎 امتیاز باقی‌مانده: " . getUserPoints($user_id),
        'file' => $file,
        'sent_message_id' => $sent_message_id
    ];
}

/**
 * Display user points
 * @param int $chat_id
 * @param int $user_id
 * @return void
 */
function displayUserPoints(int $chat_id, int $user_id): void {
    $points = getUserPoints($user_id);
    
    $message = "💎 امتیاز شما: {$points}\n\n";
    $message .= "💡 هر 1 امتیاز = 1 دانلود\n";
    $message .= "📥 برای دانلود فایل، 1 امتیاز از شما کسر می‌شود.";
    
    sendMessage($chat_id, $message);
}

/**
 * Set user points (admin only)
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function setUserPoints(int $user_id, int $points): bool {
    if ($points < 0) {
        return false;
    }
    
    // Ensure user has points record
    dbExecute(
        "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
        [$user_id]
    );
    
    $result = dbExecute(
        "UPDATE points SET points = ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

