<?php
declare(strict_types=1);

/**
 * Channel Management Handler
 * Handles forced join channel management
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Handle add channel command
 * @param array $message Telegram message object
 * @return void
 */
function handleAddChannel(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کانال اضافه کنند.");
        return;
    }
    
    // Extract channel info from message
    $channel_info = extractChannelInfo($message);
    
    if ($channel_info === false) {
        sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
        return;
    }
    
    // Check if channel already exists
    $existing = dbQueryOne(
        "SELECT id FROM channels WHERE channel_id = ?",
        [$channel_info['channel_id']]
    );
    
    if ($existing !== false) {
        sendMessage($chat_id, "❌ این کانال قبلاً اضافه شده است.");
        return;
    }
    
    // Add channel
    $result = dbExecute(
        "INSERT INTO channels (channel_id, channel_username, channel_title, channel_link) 
         VALUES (?, ?, ?, ?)",
        [
            $channel_info['channel_id'],
            $channel_info['channel_username'],
            $channel_info['channel_title'],
            $channel_info['channel_link']
        ]
    );
    
    if ($result !== false) {
        sendMessage($chat_id, "✅ کانال با موفقیت اضافه شد!");
        logInfo("Channel added: {$channel_info['channel_id']} by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در افزودن کانال.");
    }
}

/**
 * Handle remove channel command
 * @param array $message Telegram message object
 * @return void
 */
function handleRemoveChannel(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کانال حذف کنند.");
        return;
    }
    
    // Extract channel info
    $channel_info = extractChannelInfo($message);
    
    if ($channel_info === false) {
        sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
        return;
    }
    
    // Remove channel
    $result = dbExecute(
        "DELETE FROM channels WHERE channel_id = ?",
        [$channel_info['channel_id']]
    );
    
    if ($result !== false) {
        sendMessage($chat_id, "✅ کانال با موفقیت حذف شد!");
        logInfo("Channel removed: {$channel_info['channel_id']} by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در حذف کانال.");
    }
}

/**
 * Handle list channels command
 * @param int $chat_id
 * @return void
 */
function handleListChannels(int $chat_id): void {
    $channels = dbQuery("SELECT * FROM channels WHERE is_active = 1 ORDER BY created_at DESC");
    
    if ($channels === false || empty($channels)) {
        sendMessage($chat_id, "📋 لیست کانال‌های forced join خالی است.");
        return;
    }
    
    $message = "📋 لیست کانال‌های Forced Join:\n\n";
    
    foreach ($channels as $index => $channel) {
        $title = $channel['channel_title'];
        $username = $channel['channel_username'] ? '@' . $channel['channel_username'] : 'بدون یوزرنیم';
        $link = $channel['channel_link'] ?: 'بدون لینک';
        
        $message .= ($index + 1) . ". {$title}\n";
        $message .= "   📱 یوزرنیم: {$username}\n";
        $message .= "   🔗 لینک: {$link}\n";
        $message .= "   🆔 شناسه: {$channel['channel_id']}\n\n";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Extract channel info from message
 * @param array $message
 * @return array|false
 */
function extractChannelInfo(array $message): array|false {
    // Check if message is forwarded from channel
    if (isset($message['forward_from_chat'])) {
        $chat = $message['forward_from_chat'];
        
        if ($chat['type'] === 'channel') {
            return [
                'channel_id' => (int)$chat['id'],
                'channel_username' => $chat['username'] ?? null,
                'channel_title' => $chat['title'] ?? 'Unknown',
                'channel_link' => $chat['username'] ? 'https://t.me/' . $chat['username'] : null
            ];
        }
    }
    
    // Check if message text contains channel info
    if (isset($message['text'])) {
        $text = trim($message['text']);
        $text = preg_replace('/^\/\w+\s*/', '', $text);
        
        // Check if it's a username
        if (preg_match('/^@?([a-zA-Z0-9_]{5,32})$/', $text, $matches)) {
            return [
                'channel_id' => 0, // Will need to be resolved
                'channel_username' => $matches[1],
                'channel_title' => $matches[1],
                'channel_link' => 'https://t.me/' . $matches[1]
            ];
        }
        
        // Check if it's a numeric ID
        if (is_numeric($text)) {
            return [
                'channel_id' => (int)$text,
                'channel_username' => null,
                'channel_title' => 'Channel ' . $text,
                'channel_link' => null
            ];
        }
    }
    
    return false;
}

/**
 * Get all active channels
 * @return array|false
 */
function getAllActiveChannels(): array|false {
    return dbQuery("SELECT * FROM channels WHERE is_active = 1");
}

/**
 * Check if user is member of required channels
 * @param int $user_id
 * @param array|null $required_channels Array of channel IDs (if null, check all active channels)
 * @return array ['is_member' => bool, 'missing_channels' => array]
 */
function checkChannelMembership(int $user_id, ?array $required_channels = null): array {
    if ($required_channels === null) {
        $channels = getAllActiveChannels();
        if ($channels === false) {
            return ['is_member' => true, 'missing_channels' => []];
        }
        $required_channels = array_column($channels, 'channel_id');
    }
    
    $missing_channels = [];
    
    foreach ($required_channels as $channel_id) {
        // Use Telegram Bot API to check membership
        $result = bot('getChatMember', [
            'chat_id' => $channel_id,
            'user_id' => $user_id
        ]);
        
        if ($result === false || !isset($result['ok']) || !$result['ok']) {
            // If we can't check, assume user is not member
            $missing_channels[] = $channel_id;
            continue;
        }
        
        $status = $result['result']['status'] ?? 'left';
        
        // User is not a member if status is 'left' or 'kicked'
        if (in_array($status, ['left', 'kicked'], true)) {
            $missing_channels[] = $channel_id;
        }
    }
    
    return [
        'is_member' => empty($missing_channels),
        'missing_channels' => $missing_channels
    ];
}

/**
 * Set forced join channels for a file
 * @param int $file_id
 * @param array $channel_ids
 * @return bool
 */
function setFileForcedJoinChannels(int $file_id, array $channel_ids): bool {
    $channels_json = json_encode($channel_ids, JSON_UNESCAPED_UNICODE);
    
    $result = dbExecute(
        "UPDATE files SET forced_join_channels = ? WHERE id = ?",
        [$channels_json, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get forced join text from settings
 * @return string
 */
function getForcedJoinText(): string {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = 'forced_join_text'"
    );
    
    if ($setting === false) {
        return 'لطفاً ابتدا در کانال‌های زیر عضو شوید:';
    }
    
    return $setting['setting_value'] ?: 'لطفاً ابتدا در کانال‌های زیر عضو شوید:';
}

/**
 * Update forced join text
 * @param string $text
 * @return bool
 */
function updateForcedJoinText(string $text): bool {
    $result = dbExecute(
        "UPDATE settings SET setting_value = ? WHERE setting_key = 'forced_join_text'",
        [$text]
    );
    
    if ($result === false) {
        // Insert if doesn't exist
        return dbExecute(
            "INSERT INTO settings (setting_key, setting_value) VALUES ('forced_join_text', ?)",
            [$text]
        ) !== false;
    }
    
    return true;
}

