<?php
declare(strict_types=1);

/**
 * Broadcast Handler
 * Handles broadcasting messages to all users
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Broadcast normal message to all users
 * @param array $message Admin's message to broadcast
 * @param int $admin_id Admin ID
 * @return array ['success' => bool, 'sent' => int, 'failed' => int]
 */
function broadcastNormal(array $message, int $admin_id): array {
    // Check if admin
    if (!isAdmin($admin_id)) {
        return ['success' => false, 'sent' => 0, 'failed' => 0];
    }
    
    // Get all users
    $users = dbQuery("SELECT id FROM users WHERE is_blocked = 0");
    
    if ($users === false || empty($users)) {
        return ['success' => true, 'sent' => 0, 'failed' => 0];
    }
    
    $sent = 0;
    $failed = 0;
    
    // Extract message content
    $text = $message['text'] ?? null;
    $photo = $message['photo'] ?? null;
    $video = $message['video'] ?? null;
    $document = $message['document'] ?? null;
    $audio = $message['audio'] ?? null;
    $voice = $message['voice'] ?? null;
    $animation = $message['animation'] ?? null;
    $sticker = $message['sticker'] ?? null;
    
    foreach ($users as $user) {
        $user_id = (int)$user['id'];
        
        $result = false;
        
        if ($text) {
            $result = sendMessage($user_id, $text);
        } elseif ($photo) {
            $photos = is_array($photo) ? $photo : [$photo];
            $largest_photo = end($photos);
            $result = bot('sendPhoto', [
                'chat_id' => $user_id,
                'photo' => $largest_photo['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($video) {
            $result = bot('sendVideo', [
                'chat_id' => $user_id,
                'video' => $video['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($document) {
            $result = bot('sendDocument', [
                'chat_id' => $user_id,
                'document' => $document['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($audio) {
            $result = bot('sendAudio', [
                'chat_id' => $user_id,
                'audio' => $audio['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($voice) {
            $result = bot('sendVoice', [
                'chat_id' => $user_id,
                'voice' => $voice['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($animation) {
            $result = bot('sendAnimation', [
                'chat_id' => $user_id,
                'animation' => $animation['file_id'],
                'caption' => $message['caption'] ?? null
            ]);
        } elseif ($sticker) {
            $result = bot('sendSticker', [
                'chat_id' => $user_id,
                'sticker' => $sticker['file_id']
            ]);
        }
        
        if ($result !== false && isset($result['ok']) && $result['ok']) {
            $sent++;
        } else {
            $failed++;
        }
        
        // Small delay to avoid rate limiting
        usleep(50000); // 50ms
    }
    
    // Log analytics
    logAnalytics('broadcast_sent', $admin_id, null, [
        'type' => 'normal',
        'sent' => $sent,
        'failed' => $failed
    ]);
    
    return ['success' => true, 'sent' => $sent, 'failed' => $failed];
}

/**
 * Broadcast with forward
 * @param array $message Message to forward
 * @param int $admin_id Admin ID
 * @return array ['success' => bool, 'sent' => int, 'failed' => int]
 */
function broadcastForward(array $message, int $admin_id): array {
    // Check if admin
    if (!isAdmin($admin_id)) {
        return ['success' => false, 'sent' => 0, 'failed' => 0];
    }
    
    // Get all users
    $users = dbQuery("SELECT id FROM users WHERE is_blocked = 0");
    
    if ($users === false || empty($users)) {
        return ['success' => true, 'sent' => 0, 'failed' => 0];
    }
    
    $sent = 0;
    $failed = 0;
    
    $from_chat_id = $message['chat']['id'];
    $message_id = $message['message_id'];
    
    foreach ($users as $user) {
        $user_id = (int)$user['id'];
        
        $result = bot('forwardMessage', [
            'chat_id' => $user_id,
            'from_chat_id' => $from_chat_id,
            'message_id' => $message_id
        ]);
        
        if ($result !== false && isset($result['ok']) && $result['ok']) {
            $sent++;
        } else {
            $failed++;
        }
        
        // Small delay to avoid rate limiting
        usleep(50000); // 50ms
    }
    
    // Log analytics
    logAnalytics('broadcast_sent', $admin_id, null, [
        'type' => 'forward',
        'sent' => $sent,
        'failed' => $failed
    ]);
    
    return ['success' => true, 'sent' => $sent, 'failed' => $failed];
}

/**
 * Handle broadcast command
 * @param array $message Admin's message
 * @param string $type 'normal' or 'forward'
 * @return void
 */
function handleBroadcast(array $message, string $type = 'normal'): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند broadcast ارسال کنند.");
        return;
    }
    
    sendMessage($chat_id, "⏳ در حال ارسال broadcast...");
    
    if ($type === 'forward') {
        $result = broadcastForward($message, $user_id);
    } else {
        $result = broadcastNormal($message, $user_id);
    }
    
    if ($result['success']) {
        sendMessage($chat_id, "✅ Broadcast با موفقیت ارسال شد!\n📤 ارسال شده: {$result['sent']}\n❌ ناموفق: {$result['failed']}");
    } else {
        sendMessage($chat_id, "❌ خطا در ارسال broadcast.");
    }
}

