<?php
declare(strict_types=1);

final class SubscriptionService
{
    public function __construct(private readonly UserRepository $users)
    {
    }

    public function syncUser(array $telegramUser): array
    {
        $userId = (int)($telegramUser['id'] ?? 0);
        if ($userId === 0) {
            throw new \InvalidArgumentException('user id missing');
        }
        $user = $this->users->find($userId);
        if ($user) {
            return $user;
        }
        $username = $telegramUser['username'] ?? null;
        return $this->users->create([
            'id' => $userId,
            'username' => $username,
            'role' => 'user',
            'download_limit' => 3,
            'subscription_expires_at' => null,
            'keyboard_type' => 'reply',
            'theme' => 'classic',
        ]);
    }

    public function canDownload(int $userId): bool
    {
        $user = $this->users->find($userId);
        if (!$user) {
            return false;
        }
        if ($user['subscription_expires_at'] && strtotime($user['subscription_expires_at']) > time()) {
            return true;
        }
        $limit = (int)($user['download_limit'] ?? 0);
        if ($limit === 0) {
            return false;
        }
        $since = date('Y-m-d H:i:s', strtotime('-1 day'));
        $count = $this->users->downloadsCount($userId, $since);
        return $count < $limit;
    }

    public function recordDownload(int $userId, int $mediaId): void
    {
        $this->users->recordDownload($userId, $mediaId);
    }

    public function status(array $telegramUser): string
    {
        $user = $this->syncUser($telegramUser);
        $expires = $user['subscription_expires_at'];
        $limit = (int)($user['download_limit'] ?? 0);
        if ($expires && strtotime($expires) > time()) {
            return sprintf("اشتراک فعال تا %s", $expires);
        }
        return sprintf("اشتراک عادی. سقف دانلود روزانه: %d", $limit);
    }
}

