<?php
declare(strict_types=1);

use Throwable;

require_once __DIR__ . '/../../inc/keyboards.php';

final class CommandController
{
    public function __construct(
        private readonly TelegramClient $client,
        private readonly MediaService $mediaService,
        private readonly SubscriptionService $subscriptionService,
        private readonly ChannelService $channelService,
        private readonly SettingsService $settingsService,
        private readonly Database $database,
        private readonly array $appConfig
    ) {
    }

    private function crdbLockPath(): string
    {
        return project_path('storage/state/crdb.lock');
    }

    private function crdbLocked(): bool
    {
        return is_file($this->crdbLockPath());
    }

    private function lockCrdb(): void
    {
        $path = $this->crdbLockPath();
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0775, true);
        }
        file_put_contents($path, (string)time());
    }

    private function unlockCrdb(): void
    {
        $path = $this->crdbLockPath();
        if (is_file($path)) {
            unlink($path);
        }
    }

    public function start(array $message, string $payload = ''): void
    {
        $chatId = $message['chat']['id'];
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        $text = "سلام! برای آخرین انیمه‌ها از /latest و برای وضعیت از /status استفاده کن.";
        $this->sendDecoratedMessage($message, $user, $text);
        $this->sendSettingsPrompt($chatId, $user);
    }

    public function promptUpload(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'شما دسترسی آپلود ندارید.');
            return;
        }
        $text = "فایل و توضیحات را ارسال کنید تا ذخیره شود.";
        $this->client->sendMessage($message['chat']['id'], $text);
    }

    public function status(array $message, string $payload = ''): void
    {
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        $summary = $this->subscriptionService->status($message['from'] ?? []);
        $this->sendDecoratedMessage($message, $user, $summary);
    }

    public function latest(array $message, string $payload = ''): void
    {
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        $list = $this->mediaService->latestList();
        $text = $list !== '' ? $list : 'آیتمی برای نمایش وجود ندارد.';
        $this->sendDecoratedMessage($message, $user, $text);
    }

    public function getMedia(array $message, string $payload = ''): void
    {
        $mediaId = (int)trim($payload);
        if ($mediaId <= 0) {
            $this->client->sendMessage($message['chat']['id'], 'شناسه نامعتبر است.');
            return;
        }
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        if (!$this->subscriptionService->canDownload($user['id'])) {
            $this->sendDecoratedMessage($message, $user, 'سقف دانلود شما تکمیل شده است.');
            return;
        }
        if ($this->mediaService->sendMediaToUser($message['chat']['id'], $mediaId)) {
            $this->subscriptionService->recordDownload($user['id'], $mediaId);
        } else {
            $this->sendDecoratedMessage($message, $user, 'محتوا یافت نشد.');
        }
    }

    public function settings(array $message, string $payload = ''): void
    {
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        $this->sendSettingsPrompt($message['chat']['id'], $user);
    }

    public function changeTheme(array $message, string $payload = ''): void
    {
        $selection = trim($payload);
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        if ($selection === '') {
            $this->sendDecoratedMessage($message, $user, 'نام تم را بعد از دستور وارد کن. مثال: /theme luxury');
            return;
        }
        $theme = $this->settingsService->detectTheme($selection);
        if ($theme === null) {
            $this->sendDecoratedMessage($message, $user, 'تم نامعتبر است.');
            return;
        }
        $this->settingsService->setTheme($user['id'], $theme);
        $user['theme'] = $theme;
        $this->sendDecoratedMessage($message, $user, 'تم بروزرسانی شد.');
    }

    public function changeKeyboard(array $message, string $payload = ''): void
    {
        $selection = trim($payload);
        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        if ($selection === '') {
            $this->sendDecoratedMessage($message, $user, 'نوع کیبورد را وارد کن: reply یا inline');
            return;
        }
        $type = $this->settingsService->detectKeyboard($selection);
        if ($type === null) {
            $this->sendDecoratedMessage($message, $user, 'نوع کیبورد نامعتبر است.');
            return;
        }
        $this->settingsService->setKeyboard($user['id'], $type);
        $user['keyboard_type'] = $type;
        $this->sendDecoratedMessage($message, $user, 'کیبورد بروزرسانی شد.');
    }

    public function channels(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }
        $this->client->sendMessage($message['chat']['id'], $this->channelService->summary());
    }

    public function addChannel(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }
        $payload = trim($payload);
        $help = "قالب دستور: /addchannel chat_id|title|type(optional)";
        if ($payload === '') {
            $this->client->sendMessage($message['chat']['id'], $help);
            return;
        }
        $parts = array_map('trim', explode('|', $payload));
        if (count($parts) < 2) {
            $this->client->sendMessage($message['chat']['id'], $help);
            return;
        }
        $chatId = $parts[0];
        $title = $parts[1];
        $type = $parts[2] ?? 'announce';
        $this->channelService->addChannel($chatId, $title, $type);
        $this->client->sendMessage($message['chat']['id'], 'کانال ثبت شد.');
    }

    public function setDefaultChannel(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }
        $chatId = trim($payload);
        if ($chatId === '') {
            $this->client->sendMessage($message['chat']['id'], 'قالب دستور: /setchannel chat_id');
            return;
        }
        $text = $this->channelService->setDefault($chatId) ? 'کانال پیش‌فرض بروزرسانی شد.' : 'کانال یافت نشد.';
        $this->client->sendMessage($message['chat']['id'], $text);
    }

    public function publishMedia(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }
        $payload = trim($payload);
        if ($payload === '') {
            $this->client->sendMessage($message['chat']['id'], 'قالب دستور: /publish media_id [channel_id]');
            return;
        }
        $parts = preg_split('/\s+/', $payload);
        $mediaId = (int)($parts[0] ?? 0);
        $channelId = $parts[1] ?? null;
        if ($mediaId <= 0) {
            $this->client->sendMessage($message['chat']['id'], 'شناسه محتوا نامعتبر است.');
            return;
        }
        $result = $this->mediaService->publishToChannel($mediaId, $channelId);
        $this->client->sendMessage(
            $message['chat']['id'],
            $result ? 'اعلان ارسال شد.' : 'ارسال انجام نشد. شناسه را بررسی کنید.'
        );
    }

    public function createDatabase(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }

        if ($this->crdbLocked()) {
            $this->client->sendMessage($message['chat']['id'], 'این دستور قبلاً استفاده شده و غیر فعال است.');
            return;
        }

        $this->runSchema($message, true);
    }

    public function resetDatabase(array $message, string $payload = ''): void
    {
        if (!$this->isAdmin($message)) {
            $this->client->sendMessage($message['chat']['id'], 'دسترسی ندارید.');
            return;
        }
        $this->unlockCrdb();
        $this->runSchema($message, false);
    }

    private function runSchema(array $message, bool $lockAfter): void
    {
        $schemaPath = project_path('database/schema.sql');
        try {
            $this->database->runScript($schemaPath);
            if ($lockAfter) {
                $this->lockCrdb();
            }
            $this->client->sendMessage($message['chat']['id'], 'عملیات دیتابیس انجام شد.');
        } catch (Throwable $e) {
            logger('DB setup failed: ' . $e->getMessage());
            $this->client->sendMessage(
                $message['chat']['id'],
                'خطا در اجرای اسکریپت دیتابیس: ' . $e->getMessage()
            );
        }
    }

    public function handleDefault(array $message, string $payload = ''): void
    {
        if ($this->isAdmin($message) && (isset($message['video']) || isset($message['document']))) {
            $this->mediaService->handleNewUpload($message);
            return;
        }

        $user = $this->subscriptionService->syncUser($message['from'] ?? []);
        $this->sendDecoratedMessage($message, $user, 'دستور نامعتبر بود.');
    }

    public function handleThemeCallback(array $callback, string $payload): void
    {
        $userId = (int)($callback['from']['id'] ?? 0);
        if ($userId === 0 || !$this->settingsService->setTheme($userId, $payload)) {
            $this->client->answerCallback($callback['id'], 'تم نامعتبر است.');
            return;
        }
        $user = $this->subscriptionService->syncUser($callback['from']);
        $this->client->answerCallback($callback['id'], 'تم جدید فعال شد.');
        $this->refreshSettingsMarkup($callback, $user);
    }

    public function handleKeyboardCallback(array $callback, string $payload): void
    {
        $userId = (int)($callback['from']['id'] ?? 0);
        if ($userId === 0 || !$this->settingsService->setKeyboard($userId, $payload)) {
            $this->client->answerCallback($callback['id'], 'گزینه نامعتبر است.');
            return;
        }
        $user = $this->subscriptionService->syncUser($callback['from']);
        $this->client->answerCallback($callback['id'], 'کیبورد بروزرسانی شد.');
        $this->refreshSettingsMarkup($callback, $user);
    }

    public function handleActionCallback(array $callback, string $payload): void
    {
        $map = [
            'latest' => 'latest',
            'status' => 'status',
            'settings' => 'settings',
            'upload' => 'promptUpload',
        ];
        if (!isset($map[$payload])) {
            $this->client->answerCallback($callback['id'], 'ناشناخته است.');
            return;
        }
        $message = $callback['message'] ?? [
            'chat' => ['id' => $callback['from']['id']],
            'from' => $callback['from'],
        ];
        $message['from'] = $callback['from'];
        $this->{$map[$payload]}($message, '');
        $this->client->answerCallback($callback['id'], '✅');
    }

    private function isAdmin(array $message): bool
    {
        $userId = (string)($message['from']['id'] ?? '');
        return $this->isAdminId($userId);
    }

    private function isAdminId(int|string $userId): bool
    {
        return in_array((string)$userId, $this->appConfig['admin_ids'], true);
    }

    private function sendDecoratedMessage(array $message, array $user, string $text, array $options = []): void
    {
        $chatId = $message['chat']['id'];
        if (!isset($options['reply_markup'])) {
            $options['reply_markup'] = kb_build_default($user, $this->isAdminId($user['id'] ?? 0));
        }
        $formatted = $this->settingsService->decorate($text, $user['theme'] ?? 'classic');
        $this->client->sendMessage($chatId, $formatted, $options);
    }

    private function sendSettingsPrompt(int|string $chatId, array $user): void
    {
        $text = $this->settingsService->decorate($this->settingsService->summary($user), $user['theme'] ?? 'classic');
        $this->client->sendMessage($chatId, $text, [
            'reply_markup' => $this->settingsInlineKeyboard($user),
        ]);
    }

    private function settingsInlineKeyboard(array $user): array
    {
        return kb_settings_inline(
            $user,
            $this->settingsService->themeOptions(),
            $this->settingsService->keyboardOptions()
        );
    }

    private function refreshSettingsMarkup(array $callback, array $user): void
    {
        if (!isset($callback['message']['chat']['id'], $callback['message']['message_id'])) {
            return;
        }
        $this->client->editMessageReplyMarkup(
            $callback['message']['chat']['id'],
            $callback['message']['message_id'],
            $this->settingsInlineKeyboard($user)
        );
    }
}

